package com.atlassian.lesscss;

import com.atlassian.lesscss.spi.UriResolver;
import com.google.common.base.Charsets;
import com.google.common.base.Objects;
import com.google.common.io.ByteStreams;
import com.google.common.io.CharStreams;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.URI;

public class PluggableLoader implements Loader {

    private final Iterable<UriResolver> uriResolvers;

    public PluggableLoader(Iterable<UriResolver> uriResolvers) {
        this.uriResolvers = uriResolvers;
    }

    @Override
    public URI resolve(URI baseUri, String path) {
        URI resolved = baseUri.resolve(path);
        // If the schemes match, short circuit
        if (Objects.equal(baseUri.getScheme(), resolved.getScheme())) {
            return resolved;
        }

        for (UriResolver uriResolver : uriResolvers) {
            if (uriResolver.supports(resolved)) {
                return resolved;
            }
        }
        return null;
    }

    @Override
    public String load(final URI uri) throws IOException {
        for (final UriResolver uriResolver : uriResolvers) {
            if (uriResolver.supports(uri)) {
                if (!uriResolver.exists(uri)) {
                    return null;
                }
                try (Reader reader = new InputStreamReader(uriResolver.open(uri), Charsets.UTF_8)) {
                    return CharStreams.toString(reader);
                }
            }
        }
        throw new IOException("Unsupported uri: " + uri.toASCIIString());
    }

    @Override
    public String dataUri(String mimeType, final URI uri) throws IOException {
        if (mimeType == null) {
            mimeType = DataUriUtils.guessMimeType(uri.toASCIIString());
        }

        for (final UriResolver uriResolver : uriResolvers) {
            if (uriResolver.supports(uri)) {
                if (!uriResolver.exists(uri)) {
                    return null;
                }
                try (InputStream is = uriResolver.open(uri)) {
                    return DataUriUtils.dataUri(mimeType, ByteStreams.toByteArray(is));
                }
            }
        }

        throw new IOException("Unsupported uri: " + uri.toASCIIString());
    }


}
