package com.atlassian.johnson;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

import com.atlassian.johnson.event.Event;

import javax.annotation.Nonnull;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * A default implementation of {@link JohnsonEventContainer} which stores events in a list.
 * <p/>
 * Note: This implementation is thread-safe.
 *
 * @since 2.0
 */
public class DefaultJohnsonEventContainer implements JohnsonEventContainer {

    private final List<Event> events = new CopyOnWriteArrayList<Event>();

    /**
     * Adds the provided {@link Event} to the list.
     *
     * @param event the event to add
     */
    @Override
    public void addEvent(@Nonnull Event event) {
        events.add(checkNotNull(event, "event"));
    }

    /**
     * Retrieves an <i>unmodifiable</i> view of the current {@link Event} list.
     *
     * @return an unmodifiable collection of zero or more events
     */
    @Nonnull
    @Override
    public List<Event> getEvents() {
        return Collections.unmodifiableList(events);
    }

    /**
     * Retrieves a flag indicating whether there are any {@link Event}s in the list.
     *
     * @return {@link true} if the event list is not empty; otherwise, {@code false}
     */
    @Override
    public boolean hasEvents() {
        return !events.isEmpty();
    }

    /**
     * Removes the provided {@link Event} from the list, if it can be found.
     *
     * @param event the event to remove
     */
    @Override
    public void removeEvent(@Nonnull Event event) {
        events.remove(checkNotNull(event, "event"));
    }
}
