package com.atlassian.johnson;

import com.atlassian.johnson.config.ConfigurationJohnsonException;
import com.atlassian.johnson.config.DefaultJohnsonConfig;
import com.atlassian.johnson.config.JohnsonConfig;
import com.atlassian.johnson.config.XmlJohnsonConfig;
import com.atlassian.johnson.event.ApplicationEventCheck;
import com.atlassian.johnson.setup.ContainerFactory;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.ServletContext;
import java.util.List;

/**
 * Singleton class for controlling Johnson.
 * <p/>
 * Because Johnson is intended to be a maintenance and application consistency layer around an application, its
 * functionality is exposed statically. This class provides both the mechanisms for initialising and terminating
 * Johnson, as well as for retrieving its configuration and the container of events.
 * <p/>
 * Where possible to avoid synchronisation issues, it is preferable to bind Johnson to the lifecycle for the servlet
 * container, using {@link com.atlassian.johnson.context.JohnsonContextListener JohnsonContextListener} to initialise
 * and terminate Johnson.
 *
 * @since 2.0
 */
public class Johnson
{
    /**
     * When used in a web environment, the {@link JohnsonConfig} will be exposed in the {@code ServletContext} under
     * an attribute with this key.
     */
    public static final String CONFIG_ATTRIBUTE = Johnson.class.getName() + ":Config";
    /**
     * During {@link #initialize(javax.servlet.ServletContext) initialisation}, the {@code ServletContext} is examined
     * for an {@code init-param} with this name. If one is found, it controls the location from which the configuration
     * is loaded. Otherwise, {@link XmlJohnsonConfig#DEFAULT_CONFIGURATION_FILE} is used as the default.
     */
    public static final String CONFIG_LOCATION_PARAM = "johnsonConfigLocation";
    /**
     * When used in a web environment, the {@link JohnsonEventContainer} will be exposed in the {@code ServletContext}
     * under an attribute with this key.
     */
    public static final String EVENT_CONTAINER_ATTRIBUTE = Johnson.class.getName() + ":EventContainer";

    private static final Logger LOG = LoggerFactory.getLogger(Johnson.class);

    private static JohnsonConfig config;
    private static JohnsonEventContainer eventContainer;

    private Johnson()
    {

    }

    /**
     * Retrieves the statically-bound {@link JohnsonConfig}.
     * <p/>
     * Note: If Johnson has not been {@link #initialize(String) initialised}, this method <i>will not</i> initialise
     * it. Before attempting to use Johnson, it is left to the application developer to ensure it has been correctly
     * initialised.
     *
     * @return the working configuration
     * @throws IllegalStateException if initialize has not been called
     */
    public static JohnsonConfig getConfig()
    {
        if (config == null)
        {
            throw new IllegalStateException("Johnson.getConfig() was called before initialisation");
        }
        return config;
    }

    /**
     * Attempts to retrieve the {@link JohnsonConfig} from the provided {@code ServletContext} under the key
     * {@link #CONFIG_ATTRIBUTE} before falling back on {@link #getConfig()}.
     * <p/>
     * Note: If Johnson has not been {@link #initialize(ServletContext) initialised}, this method <i>will not</i>
     * initialise it. Before attempting to use Johnson, it is left to the application developer to ensure it has
     * been correctly initialised.
     *
     * @param context the servlet context
     * @return the working configuration
     */
    public static JohnsonConfig getConfig(ServletContext context)
    {
        Object attribute = context.getAttribute(CONFIG_ATTRIBUTE);
        if (attribute != null)
        {
            return (JohnsonConfig) attribute;
        }
        return config;
    }

    /**
     * Attempts to retrieve the {@link JohnsonEventContainer} from the provided {@code ServletContext} under the key
     * {@link #EVENT_CONTAINER_ATTRIBUTE} before falling back on the statically-bound instance.
     * <p/>
     * Note: If Johnson has not been {@link #initialize(ServletContext) initialised}, this method <i>will not</i>
     * initialise it. Before attempting to use Johnson, it is left to the application developer to ensure it has
     * been correctly initialised.
     *
     * @param context the servlet context
     * @return the working event container
     */
    public static JohnsonEventContainer getEventContainer(ServletContext context)
    {
        Object attribute = context.getAttribute(EVENT_CONTAINER_ATTRIBUTE);
        if (attribute != null)
        {
            return (JohnsonEventContainer) attribute;
        }
        return eventContainer;
    }

    /**
     * Initialises Johnson, additionally binding the {@link JohnsonConfig} and {@link JohnsonEventContainer} to the
     * provided {@code ServletContext} and performing any {@link ApplicationEventCheck}s which have been configured.
     * <p/>
     * If the {@link #CONFIG_LOCATION_PARAM} {@code init-param} is set, it is used to determine the location of the
     * Johnson configuration file. Otherwise, {@link XmlJohnsonConfig#DEFAULT_CONFIGURATION_FILE} is assumed.
     * <p/>
     * Note: This method is <i>not synchronised</i> and <i>not thread-safe</i>. It is left to the <i>calling code</i>
     * to ensure proper synchronisation. The easiest way to do this is to initialise Johnson by adding the
     * {@link com.atlassian.johnson.context.JohnsonContextListener} to {@code web.xml}.
     *
     * @param context the servlet context
     */
    public static void initialize(ServletContext context)
    {
        String location = StringUtils.defaultIfEmpty(context.getInitParameter(CONFIG_LOCATION_PARAM),
                XmlJohnsonConfig.DEFAULT_CONFIGURATION_FILE);

        initialize(location);

        context.setAttribute(CONFIG_ATTRIBUTE, config);
        context.setAttribute(EVENT_CONTAINER_ATTRIBUTE, eventContainer);

        List<ApplicationEventCheck> checks = config.getApplicationEventChecks();
        for (ApplicationEventCheck check : checks)
        {
            check.check(eventContainer, context);
        }
    }

    /**
     * Initialises Johnson, loading its configuration from the provided location, and sets the statically-bound
     * instances of {@link JohnsonConfig} and {@link JohnsonEventContainer}.
     * <p/>
     * If the location provided is {@code null} or empty, {@link XmlJohnsonConfig#DEFAULT_CONFIGURATION_FILE} is
     * assumed.
     * <p/>
     * Note: If the configuration fails to load, {@link DefaultJohnsonConfig} is used to provide defaults. For more
     * information about what those defaults are, see the documentation for that class.
     *
     * @param location the location of the Johnson configuration file
     */
    public static void initialize(String location)
    {
        location = StringUtils.defaultIfEmpty(location, XmlJohnsonConfig.DEFAULT_CONFIGURATION_FILE);

        LOG.debug("Initialising Johnson with configuration from [{}]", location);
        try
        {
            config = XmlJohnsonConfig.fromFile(location);
        }
        catch (ConfigurationJohnsonException e)
        {
            LOG.warn("Failed to load configuration from [" + location + "]", e);
            config = DefaultJohnsonConfig.getInstance();
        }

        ContainerFactory containerFactory = config.getContainerFactory();
        eventContainer = containerFactory.create();
    }

    /**
     * Terminates Johnson, clearing the statically-bound {@link JohnsonConfig} and {@link JohnsonEventContainer}.
     */
    public static void terminate()
    {
        config = null;
        eventContainer = null;
    }

    /**
     * Terminates Johnson, removing Johnson-related attributes from the provided {@code ServletContext} and clearing
     * the statically-bound {@link JohnsonConfig} and {@link JohnsonEventContainer}.
     *
     * @param context the servlet context
     */
    public static void terminate(ServletContext context)
    {
        terminate();

        context.removeAttribute(CONFIG_ATTRIBUTE);
        context.removeAttribute(EVENT_CONTAINER_ATTRIBUTE);
    }
}
