package com.atlassian.johnson;

import com.atlassian.johnson.event.Event;

import javax.servlet.ServletContext;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

/**
 * A default implementation of {@link JohnsonEventContainer} which stores events in a list.
 * <p/>
 * Note: This implementation is thread-safe.
 *
 * @since 2.0
 */
public class DefaultJohnsonEventContainer implements JohnsonEventContainer
{
    private final List<Event> events = new CopyOnWriteArrayList<Event>();

    /**
     * Adds the provided {@link Event} to the list.
     * 
     * @param event the event to add
     */
    @Override
    public void addEvent(Event event)
    {
        events.add(event);
    }

    /**
     * Retrieves an <i>unmodifiable</i> view of the current {@link Event} list.
     * 
     * @return an unmodifiable collection of zero or more events
     */
    @Override
    public List<Event> getEvents()
    {
        return Collections.unmodifiableList(events);
    }

    /**
     * Retrieves a flag indicating whether there are any {@link Event}s in the list.
     * 
     * @return {@link true} if the event list is not empty; otherwise, {@code false}
     */
    @Override
    public boolean hasEvents()
    {
        return !events.isEmpty();
    }

    /**
     * Removes the provided {@link Event} from the list, if it can be found.
     *
     * @param event the event to remove
     */
    @Override
    public void removeEvent(Event event)
    {
        events.remove(event);
    }
}
