(function () {
    var highlightedColour = "#707070",
        maxLabelLength = 30,
        normalColour = "#cccccc",
        normalLabelColour = "#ffffff",
        selectedColour = "#1a8cff",
        selectedLabelColour = "#f5f5f5";

    AJS.namespace("JIRA.WorkflowDesigner.TransitionConnection");

    JIRA.WorkflowDesigner.TransitionConnection = JIRA.WorkflowDesigner.Layout.Connection.GappedConnection.extend(
    /** @lends JIRA.WorkflowDesigner.TransitionConnection# */
    {
        /**
         * Initialise the connection.
         *
         * @constructs
         * @extends JIRA.WorkflowDesigner.Layout.Connection.GappedConnection
         * @classdesc `TransitionView`'s Draw2D connection.
         * @param {boolean} isInitialTransition Whether this connection represents the initial transition.
         */
        init: function (isInitialTransition) {
            var arrow, label;

            _.bindAll(this, "_onDoubleClick", "_onMouseEnter", "_onMouseLeave");

            this._isInitialTransition = isInitialTransition;
            if (isInitialTransition) {
                this._super(0,3);
            } else {
                this._super(3,3);
            }

            arrow = new draw2d.decoration.connection.ArrowDecorator();
            arrow.onDoubleClick = this._onDoubleClick;
            arrow.setBackgroundColor(normalColour);
            arrow.setColor(normalColour);
            arrow.setDimension(7, 7);

            label = this._label = new draw2d.shape.basic.Label();
            label.onDoubleClick = this._onDoubleClick;
            label.onMouseEnter = this._onMouseEnter;
            label.onMouseLeave = this._onMouseLeave;
            label.setBackgroundColor(normalLabelColour);
            label.setColor("#cccccc");
            label.setBold(true);
            label.setDraggable(false);
            label.setFontColor("#333333");
            label.setFontFamily("Arial");
            label.setFontSize(11);
            label.setPadding(2);
            label.setParent(this);
            label.setRadius(3);
            label.setSelectable(false);
            label.setStroke(0);

            this.setColor(normalColour);
            this.setCoronaWidth(6);
            this.setDeleteable(false);
            this.setResizeable(false);
            this.setRouter(new JIRA.WorkflowDesigner.Layout.Connection.RoundedManhattanConnectionRouter());
            this.setStroke(1);
            this.setTargetDecorator(arrow);
        },

        /**
         * Make the connection appear selected.
         */
        appearSelected: function () {
            this.setColor(selectedColour);
            this.targetDecoratorNode.attr({
                fill: "#ffffff",
                stroke: selectedColour
            }).insertAfter(this.shape);

            this._label.setBackgroundColor(selectedLabelColour).setStroke(1);
        },

        /**
         * Make the connection appear highlighted.
         */
        highlight: function () {
            this.setColor(highlightedColour);
            this.targetDecoratorNode.attr({
                fill: highlightedColour,
                stroke: highlightedColour
            }).insertAfter(this.shape);
        },

        /**
         * Called when the connection is double clicked.
         *
         * @private
         */
        _onDoubleClick: function () {
            this.onDoubleClick.apply(this, arguments);
        },

        /**
         * Called when the mouse enters the connection's label.
         *
         * @private
         */
        _onMouseEnter: function () {
            this.onMouseEnter.apply(this, arguments);
        },

        /**
         * Called when the mouse leaves the connection's label.
         *
         * @private
         */
        _onMouseLeave: function () {
            this.onMouseLeave.apply(this, arguments);
        },

        /**
         * Position the connection's arrow (including z-index).
         */
        positionArrow: function () {
            this.targetDecoratorNode.insertAfter(this.shape);
        },

        /**
         * Position the connection's label.
         *
         * @private
         */
        _positionLabel: function () {
            this._labelLocator || (this._labelLocator = new JIRA.WorkflowDesigner.Layout.Locator.ManhattanPercentageLocator(this, 0.8));
            this._labelLocator.relocate(0, this._label);
        },

        /**
         * Make the connection appear selected.
         *
         * @return {JIRA.WorkflowDesigner.TransitionConnection} `this`
         */
        select: function () {
            var startSelectionHandle;
            this._super.apply(this, arguments);

            this.appearSelected();

            // Selection handles must be above the connection's line and we
            // must hide the start selection handle of the initial transition.
            startSelectionHandle = this.selectionHandles.get(0);
            this._isInitialTransition && startSelectionHandle.hide();
            startSelectionHandle.sendToFront();

            // Selection handles' coronas must be on front, as they handle mouse
            // interactions and modify the mouse cursor
            this.selectionHandles.get(0).sendActionableAreaToFront();
            this.selectionHandles.get(1).sendActionableAreaToFront();

            return this;
        },

        /**
         * Set the connection's canvas.
         *
         * @param {draw2d.Canvas} canvas The canvas.
         */
        setCanvas: function (canvas) {
            var isAdding = canvas !== null,
                isRemoving = canvas === null;

            if (isAdding) {
                canvas.getLayer("transition-labels").addFigure(this._label);
            } else if (isRemoving) {
                this.canvas.removeFigure(this._label);
            }

            this._super(canvas);
        },

        /**
         * Set the connection's text.
         *
         * @param {string} [text] The connection's text.
         */
        setText: function (text) {
            this._label.setText(this._truncate(text));
            this._positionLabel();
        },

        /**
         * Truncates the text to the maximum allowed number of characters.
         * If text length is less than maximum then returns the original text.
         * Otherwise truncates the text and appends ellipsis to it.
         *
         * @param [text] Text to truncate
         * @returns {string} truncated text
         * @private
         */
        _truncate: function(text) {
            var ellipsis = this._getEllipsis();

            if (!text || text.length <= maxLabelLength) {
                return text;
            }

            return text.substring(0, maxLabelLength - ellipsis.length) + ellipsis;
        },

        /**
         * @returns {string}
         * @private
         */
        _getEllipsis: function() {
            return AJS.I18n.getText("workflow.designer.ellipsis");
        },

        /**
         * Set the visibility of the connection's text.
         *
         * @param {boolean} visible Whether the connection's text should be visible.
         */
        setTextVisible: function (visible) {
            this._label.setVisible(visible);
            this._positionLabel();
        },

        /**
         * Sets the source port of the connection.
         * @param {draw2d.Port} port The new source port of this connection.
         */
        setSource: function(port) {
            this._super(port);
            this._positionLabel();
        },

        /**
         * Sets the target port of the connection.
         * @param {draw2d.Port} port The new target port of this connection
         */
        setTarget: function(port) {
            this._super(port);
            this._positionLabel();
        },

        /**
         * @return {boolean} Whether the connection's text is visible.
         */
        textIsVisible: function () {
            return this._label.isVisible();
        },

        /**
         * Make the connection appear unhighlighted.
         */
        unhighlight: function () {
            this._label.setBackgroundColor(normalLabelColour).setStroke(0);
            this.setColor(normalColour);
            this.targetDecoratorNode.attr({
                fill: normalColour,
                stroke: normalColour
            }).insertAfter(this.shape);
        },

        /**
         * Make the connection appear unselected.
         *
         * @return {JIRA.WorkflowDesigner.TransitionConnection} `this`
         */
        unselect: function () {
            this._super.apply(this, arguments);
            this.unhighlight();
            this._label.setBackgroundColor(normalLabelColour).setStroke(0);
            return this;
        }
    });
}());