(function () {
    AJS.namespace("JIRA.WorkflowDesigner.StatusFigure");

    JIRA.WorkflowDesigner.StatusFigure = draw2d.shape.basic.Rectangle.extend({
        /**
         * @classdesc {@link JIRA.WorkflowDesigner.StatusView}'s Draw2D figure.
         * @constructs
         * @extends draw2d.shape.basic.Rectangle
         */
        init: function () {
            var edge, fill, hover, label, locator, selectedOutline;

            locator = new draw2d.layout.locator.CenterLocator(this);
            this._super.apply(this, arguments);

            this.installEditPolicy(new draw2d.policy.figure.SelectionFeedbackPolicy());
            this.setAlpha(0);
            this.setDeleteable(false);
            this.setStroke(0);
            this.snapToGeometry = true;

            // The hover figure intercepts the status figure's events and is
            // larger than the visible status area, simplifying interaction.
            hover = this._hover = new draw2d.shape.basic.Rectangle();
            hover.onDoubleClick = _.bind(this._onDoubleClick, this);
            hover.onMouseEnter = _.bind(this._onMouseEnter, this);
            hover.onMouseLeave = _.bind(this._onMouseLeave, this);
            hover.setAlpha(0);
            this.addFigure(hover, locator);

            edge = this._edge = new draw2d.shape.basic.Rectangle();
            edge.setRadius(4);
            edge.setStroke(0);
            this.addFigure(edge, locator);

            // The highlight shown on the selected status.
            selectedOutline = this._selectedOutline = new draw2d.shape.basic.Rectangle();
            selectedOutline.setColor("#1a8cff");
            selectedOutline.setRadius(4);
            selectedOutline.setStroke(2);
            selectedOutline.setVisible(false);
            this.addFigure(selectedOutline, locator);

            fill = this._fill = new draw2d.shape.basic.Rectangle();
            fill.setRadius(3);
            fill.setStroke(0);
            this.addFigure(fill, locator);

            label = this._label = new draw2d.shape.basic.Label();
            label.setBold(true);
            label.setFontFamily("Arial");
            label.setFontSize(11);
            label.setStroke(0);
            this.addFigure(label, locator);
        },

        /**
         * Get the figure's text colour.
         *
         * @return {draw2d.util.Color} A colour.
         */
        getTextColor: function () {
            return this._label.getFontColor();
        },

        /**
         * Call the figure's `onDoubleClick` handler.
         *
         * @private
         */
        _onDoubleClick: function () {
            this.onDoubleClick.apply(this, arguments);
        },

        /**
         * Call the figure's `onMouseEnter` handler.
         *
         * @private
         */
        _onMouseEnter: function () {
            this.onMouseEnter.apply(this, arguments);
        },

        /**
         * Call the figure's `onMouseLeave` handler.
         *
         * @private
         */
        _onMouseLeave: function () {
            this.onMouseLeave.apply(this, arguments);
        },

        /**
         * Repaint the figure.
         */
        repaint: function () {
            function setCursor (child) {
                JIRA.WorkflowDesigner.Draw2DUtilities.setCursor(child.figure, "move");
            }

            this._super.apply(this, arguments);
            _.each(this.children.asArray(), setCursor);
        },

        /**
         * Calculate and set the figure's dimensions.
         *
         * @private
         */
        _resize: function () {
            var width = Math.max(this._label.getBoundingBox().getWidth() + 40, 60);
            this.setDimension(width, 30);
        },

        /**
         * @method
         */
        select: function () {
            this._super.apply(this, arguments);
            this._selectedOutline.setVisible(true);
        },

        /**
         * Set the figure's background colour.
         *
         * @param {string} colour A colour.
         */
        setBackgroundColor: function (colour) {
            this._super.apply(this, arguments);
            this._fill && this._fill.setBackgroundColor(colour);
        },

        /**
         * Set the figure's canvas.
         *
         * @param {JIRA.WorkflowDesigner.Draw2DCanvas} canvas A Draw2D canvas.
         */
        setCanvas: function (canvas) {
            this._super.apply(this, arguments);

            // This must be done after the shape has been added to the canvas.
            canvas && this._selectedOutline.shape.attr("fill-opacity", "0");
        },

        /**
         * Set the figure's stroke colour.
         *
         * @param {string} colour A colour.
         */
        setColor: function (colour) {
            this._super.apply(this, arguments);
            this._edge && this._edge.setBackgroundColor(colour);
        },

        /**
         * Set the figure's dimensions.
         *
         * @param {number} width The figure's width.
         * @param {number} height The figure's height.
         */
        setDimension: function (width, height) {
            var isInitialised = !!this._edge;
            this._super.apply(this, arguments);

            if (isInitialised) {
                this._edge.setDimension(width, height);
                this._fill.setDimension(width - 2, height - 2);
                this._hover.setDimension(width + 12, height + 12);
                this._selectedOutline.setDimension(width + 1, height + 1);
            }
        },

        /**
         * Set the figure's text.
         *
         * @param {string} text The figure's text.
         */
        setText: function (text)  {
            this._label.setText(text.toUpperCase());
            this._resize();
        },

        /**
         * Set the figure's text colour.
         *
         * @param {string} colour A colour.
         */
        setTextColor: function (colour) {
            this._label && this._label.setFontColor(colour);
        },

        /**
         * @method
         */
        unselect: function () {
            this._super.apply(this, arguments);
            this._selectedOutline.setVisible(false);
        }
    });
}());