(function (Backbone) {
    var AddTransitionFormView = JIRA.WorkflowDesigner.AddTransitionFormView;

    Backbone.define("JIRA.WorkflowDesigner.NewTransitionFormView", AddTransitionFormView.extend({
        template: JIRA.WorkflowDesigner.Templates.newTransitionForm,

        ui: {
            description: "#transition-description",
            name: "#transition-name",
            screenId: "#transition-screen-id",
            sourceStepId: "#transition-source-step-id",
            targetStepId: "#transition-target-step-id"
        },

        /**
         * Initialize the view.
         *
         * @constructs
         * @classdesc A form for creating new transitions.
         *
         * This view must be shown in a `Marionette.Region` to work correctly.
         * @extends JIRA.WorkflowDesigner.AddTransitionFormView
         * @param {object} options
         * @param {object[]} options.screens The screens that can be selected.
         * @param {object[]} options.sourceStatuses The source statuses that can be selected.
         * @param {object[]} options.targetStatuses The target statuses that can be selected.
         * @param {JIRA.WorkflowDesigner.WorkflowModel} options.workflowModel The application's workflow model.
         */
        initialize: function () {
            _.bindAll(this, "_onSuccess");
        },

        /**
         * @returns {number} The ID of the selected screen.
         * @private
         */
        _getSelectedScreenId: function () {
            return Number(this.ui.screenId.find(":selected").val());
        },

        /**
         * @returns {string} The name of the selected screen.
         * @private
         */
        _getSelectedScreenName: function () {
            return this.ui.screenId.find(":selected").text();
        },

        onShow: function () {
            AddTransitionFormView.prototype.onShow.apply(this, arguments);

            new AJS.SingleSelect({
                element: this.ui.screenId,
                revertOnInvalid: true
            });
        },

        /**
         * @param {object} response An AJAX response.
         * @private
         */
        _onSuccess: function (response) {
            var attributes;

            attributes = _.extend({}, response, {
                description: this.ui.description.val(),
                name: this.ui.name.val(),
                screenId: this._getSelectedScreenId(),
                screenName: this._getSelectedScreenName(),
                source: this._getSelectedSourceStatus(),
                target: this._getSelectedTargetStatus()
            });

            this.model.set(attributes);
        },

        serializeData: function () {
            return {
                screens: this.options.screens,
                sourceStatuses: this.options.sourceStatuses,
                targetStatuses: this.options.targetStatuses
            };
        },

        /**
         * Attempt to create a new transition.
         *
         * @returns {jQuery.Deferred} A promise that is resolved on completion.
         */
        submit: function () {
            var deferred = jQuery.Deferred(),
                options,
                request;

            options = {
                description: this.ui.description.val(),
                name: this.ui.name.val(),
                screenId: this._getSelectedScreenId(),
                sourceStepId: this._getSelectedSourceStatus().get("stepId"),
                targetStepId: this._getSelectedTargetStatus().get("stepId"),
                workflowName: this.options.workflowModel.get("name")
            };

            request = JIRA.WorkflowDesigner.IO.AJAX.WorkflowTransitionsAJAXManager.addTransition(options);
            request.done(this._onSuccess).done(deferred.resolve).fail(deferred.reject);
            return deferred.promise();
        }
    }));
}(JIRA.WorkflowDesigner.Backbone));