AJS.test.require("com.atlassian.jira.plugins.jira-workflow-designer:test-resources");
AJS.test.require("com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer");

module("JIRA.WorkflowDesigner.Dialogs.EditStatusDialogView", {
    setup: function () {
        var getStatusesDeferred,
            sandbox = this.sandbox = sinon.sandbox.create();

        getStatusesDeferred = jQuery.Deferred().resolve([{
            id: 100,
            numberOfWorkflows: 1
        }]);

        this.getAssociatedWorkflowsStub = sandbox.stub(JIRA.WorkflowDesigner.IO.AJAX.StatusesAJAXManager, "getNumberOfAssociatedWorkflows").returns(5);
        this.getStatusesStub = sandbox.stub(JIRA.WorkflowDesigner.IO.AJAX.StatusesAJAXManager, "getStatuses").returns(getStatusesDeferred);
        this.updateStatusStub = sandbox.stub(JIRA.WorkflowDesigner.IO.AJAX.WorkflowStatusesAJAXManager, "updateStatus").returns(jQuery.Deferred());
    },

    /**
     * Create and show an `EditStatusDialogView`.
     *
     * @param {object} [options] Options to pass to the dialog's constructor.
     * @return {object} The dialog's element.
     */
    showDialog: function (options) {
        options = _.defaults({}, options, {
            statusModel: new JIRA.WorkflowDesigner.StatusModel({
                statusCategory: {
                    id: 3
                },
                statusId: 100,
                stepId: 1,
                name: "Status",
                description: "This is the description."
            }),
            workflowModel: new JIRA.WorkflowDesigner.WorkflowModel({
                name: "Workflow Name"
            })
        });

        new JIRA.WorkflowDesigner.Dialogs.EditStatusDialogView(options).show();
        return JIRA.Dialog.current && JIRA.Dialog.current.get$popup();
    },

    teardown: function () {
        JIRA.WorkflowDesigner.TestUtilities.removeDialogs();
        this.sandbox.restore();
    }
});

test("A warning is shown if the status is used in other workflows", function () {
    this.getStatusesStub.returns(jQuery.Deferred().resolve([{
        id: 100
    }]));

    equal(this.showDialog().find(".aui-message.warning").length, 1, "A warning is shown");
});

test("An error message is shown if editing the status fails", function () {
    var element = this.showDialog(),
        errorElement,
        errorMessage = "Oh noes!";

    this.updateStatusStub.returns(jQuery.Deferred().reject(errorMessage).promise());
    JIRA.WorkflowDesigner.TestUtilities.submit(element);

    errorElement = element.find(".aui-message.error");
    equal(errorElement.length, 1, "An error message is shown in the dialog");
    ok(errorElement.text().indexOf(errorMessage) > -1, "It contains the correct text");
    equal(element.find(":input:disabled").length, 0, "No elements in the dialog are disabled");

    this.updateStatusStub.returns(jQuery.Deferred().promise());
    JIRA.WorkflowDesigner.TestUtilities.submit(element);
    equal(element.find(".error").length, 0, "Errors are removed on submit");
});

test("An error message is shown if loading associated workflows fails", function () {
    var errorMessage = "No associated workflows for you!",
        showErrorMessageStub = this.sandbox.stub(JIRA.WorkflowDesigner.Messages, "showErrorMessage");

    this.getAssociatedWorkflowsStub.returns(jQuery.Deferred().reject(errorMessage));

    this.showDialog();
    ok(!JIRA.WorkflowDesigner.TestUtilities.dialogIsVisible(), "The dialog was hidden");
    ok(showErrorMessageStub.calledWithExactly(errorMessage), "An error message was shown");
});

test("An error message is shown if loading statuses fails", function () {
    var errorMessage = "No statuses for you!",
        showErrorMessageStub = this.sandbox.stub(JIRA.WorkflowDesigner.Messages, "showErrorMessage");

    this.getStatusesStub.returns(jQuery.Deferred().reject(errorMessage));

    this.showDialog();
    ok(!JIRA.WorkflowDesigner.TestUtilities.dialogIsVisible(), "The dialog was hidden");
    ok(showErrorMessageStub.calledWithExactly(errorMessage), "An error message was shown");
});

test("Editing a status", function () {
    var deferred = jQuery.Deferred(),
        element,
        expectedAttributes,
        expectedArguments,
        statusModel,
        statusModelSpy = this.sandbox.spy(JIRA.WorkflowDesigner, "StatusModel");

    element = this.showDialog();
    statusModel = statusModelSpy.thisValues[0];

    this.updateStatusStub.returns(deferred.promise());
    this.sandbox.stub(JIRA.WorkflowDesigner.Analytics, "triggerUpdateStatus");
    this.sandbox.stub(JIRA.WorkflowDesigner.Analytics, "triggerUpdateStep");

    JIRA.WorkflowDesigner.TestUtilities.submit(element, {
        "#status-description": "Awesome new description!",
        "#status-name": "New Name"
    });

    equal(element.find(":input:disabled").length, element.find(":input").length, "All inputs are disabled while submitting");
    equal(this.updateStatusStub.callCount, 1, "WorkflowStatusesAJAXManager.updateStatus() was called");

    deferred.resolve();
    expectedArguments = [{
        description: "Awesome new description!",
        name: "New Name",
        statusCategoryId: undefined,
        statusId: 100,
        workflowName: "Workflow Name"
    }];
    expectedAttributes = {
        description: "Awesome new description!",
        name: "New Name",
        statusCategory: null
    };

    deepEqual(this.updateStatusStub.args[0], expectedArguments, "It was passed the correct arguments");
    deepEqual(_.pick(statusModel.attributes, _.keys(expectedAttributes)), expectedAttributes,
            "The StatusModel's attributes were updated");

    ok(JIRA.WorkflowDesigner.Analytics.triggerUpdateStatus.calledOnce, "Analytics event for update/status has been triggered");
    ok(JIRA.WorkflowDesigner.Analytics.triggerUpdateStep.calledOnce, "Analytics event for udpate/step has been triggered");
});

test("Fields are populated with the status's attributes", function () {
    var element = this.showDialog();

    equal(element.find("#status-description").val(), "This is the description.", "The status's description was used");
    equal(element.find("#status-name").val(), "Status", "The status's name was used");
});

test("No warning is shown if the status isn't used in other workflows", function () {
    this.getAssociatedWorkflowsStub.returns(jQuery.Deferred().resolve(0));

    equal(this.showDialog().find(".aui-message.warning").length, 0, "No warning is visible");
});

test("The selected status category is used", function () {
    var statusCategories,
        StatusCategoriesAJAXManager = JIRA.WorkflowDesigner.IO.AJAX.StatusCategoriesAJAXManager,
        statusModel,
        statusModelSpy = this.sandbox.spy(JIRA.WorkflowDesigner, "StatusModel");

    statusCategories = [{
        colorName: "blue-gray",
        id: 2,
        key: "new",
        name: "New"
    }, {
        colorName: "green",
        id: 3,
        key: "done",
        name: "Complete"
    }];

    this.sandbox.stub(StatusCategoriesAJAXManager, "getStatusCategories").returns(jQuery.Deferred().resolve(statusCategories));
    this.updateStatusStub.returns(jQuery.Deferred().resolve());

    JIRA.WorkflowDesigner.TestUtilities.withDarkFeature("jira.issue.status.lozenge", true, function () {
        JIRA.WorkflowDesigner.TestUtilities.submit(this.showDialog(), {
            "#status-category-id": "2"
        });

        equal(this.updateStatusStub.callCount, 1, "WorkflowStatusesAJAXManager#updateStatus was called");
        equal(this.updateStatusStub.args[0][0].statusCategoryId, "2", "It was passed the correct status category");

        statusModel = statusModelSpy.thisValues[0];
        deepEqual(statusModel.get("statusCategory"), {colourName: "blue-gray", id: 2},
                "The StatusModel's statusCategory attribute was updated");
    }, this);
});