(function () {
    /**
     * Insert one figure after another.
     *
     * @param {draw2d.Figure} figure The figure to insert.
     * @param {draw2d.Figure} target The figure to insert after.
     */
    function insertFigure(figure, target) {
        var children = figure.getChildren().clone().asArray().reverse(),
            targetNode = target.svgNodes || target.shape;

        _.each(children, function (child) {
            insertFigure(child, target);
        });

        // This order is important; svgNodes must always come after shape.
        figure.svgNodes && figure.svgNodes.insertAfter(targetNode);
        figure.shape && figure.shape.insertAfter(targetNode);
    }

    AJS.namespace("JIRA.WorkflowDesigner.Draw2DCanvas.Layer");

    /**
     * A layer of figures in a canvas.
     *
     * @class
     * @param {JIRA.WorkflowDesigner.Draw2DCanvas} canvas A canvas.
     */
    JIRA.WorkflowDesigner.Draw2DCanvas.Layer = function (canvas) {
        var placeholder = new JIRA.WorkflowDesigner.Draw2DCanvas.LayerRootFigure();

        this._canvas = canvas;
        this._canvas.addFigure(placeholder);
        this._figures = [placeholder];
    };

    _.extend(JIRA.WorkflowDesigner.Draw2DCanvas.Layer.prototype,
    /** @lends JIRA.WorkflowDesigner.Draw2DCanvas.Layer# */
    {
        /**
         * Add a figure to the layer.
         *
         * Adds the figure to the canvas if necessary and removes it from its current layer.
         *
         * @param {draw2d.Figure} figure The figure.
         */
        addFigure: function (figure) {
            figure.getCanvas() || this._canvas.addFigure(figure);
            figure._layer && figure._layer.removeFigure(figure);

            figure._layer = this;
            insertFigure(figure, _.last(this._figures));
            this._figures.push(figure);
        },

        /**
         * Remove a figure from the layer.
         *
         * This doesn't remove the figure from the canvas or reposition it.
         *
         * @param {draw2d.Figure} figure The figure.
         */
        removeFigure: function (figure) {
            this._figures = _.without(this._figures, figure);
        }
    });
}());