(function () {
    AJS.namespace("JIRA.WorkflowDesigner.Draw2DCanvas");

    JIRA.WorkflowDesigner.Draw2DCanvas = draw2d.Canvas.extend(
    /** @lends JIRA.WorkflowDesigner.Draw2DCanvas# */
    {
        /**
         * Initialise the canvas.
         *
         * @constructs
         * @extends draw2d.Canvas
         */
        init: function () {
            _.bindAll(this, "_scheduleResize");

            this._super.apply(this, arguments);

            this.paper.canvas.style.position = ""; // Draw2D doesn't like Raphael's sub-pixel rendering fix; kill it.
            this.svgElement = jQuery(this.paper.canvas);

            this._debouncedSetViewBox = _.debounce(this.setViewBox, 100);

            this._scheduleResize();
        },

        /**
         * Create one or more named layers.
         */
        createLayers: function () {
            var Layer = JIRA.WorkflowDesigner.Draw2DCanvas.Layer,
                layers = this._layers || (this._layers = {});

            _.each(_.flatten(arguments), function (name) {
                layers[name] || (layers[name] = new Layer(this));
            }, this);
        },

        /**
         * Resize the SVG element to fill its container.
         */
        fitToContainer: function () {
            var canvasHeight = Number(this.svgElement.attr("height")),
                canvasWidth = Number(this.svgElement.attr("width")),
                containerHeight = this.getHeight(),
                containerWidth = this.getWidth(),
                shouldFit = canvasHeight !== containerHeight || canvasWidth !== containerWidth;

            if (!this.html.is(":visible")) {
                return;
            }

            if (shouldFit) {
                this.paper.setSize(containerWidth, containerHeight);
                this.svgElement.css({
                    height: containerHeight,
                    width: containerWidth
                });
            }
        },

        /**
         * Convert a point from canvas space to document space.
         *
         * The returned point is relative to the document's frame,
         * not its root element (i.e. scrolling won't change it).
         *
         * @param {number} x The point's x coordinate in canvas space.
         * @param {number} y The point's y coordinate in canvas space.
         * @return {draw2d.geo.Point} The point in document space.
         */
        fromCanvasToDocumentCoordinate: function (x, y) {
            var point = JIRA.WorkflowDesigner.SVGUtilities.fromSVGToScreenCoordinate(this.paper.canvas, x, y);
            return new draw2d.geo.Point(point.x, point.y);
        },

        /**
         * Convert a point from document space to canvas space.
         *
         * The given point should be relative to the documents frame,
         * not its root element (i.e. scrolling shouldn't change it).
         *
         * @param {number} x The point's x coordinate in document space.
         * @param {number} y The point's y coordinate in document space.
         * @return {draw2d.geo.Point} The point in canvas space.
         */
        fromDocumentToCanvasCoordinate: function (x, y) {
            var point = JIRA.WorkflowDesigner.SVGUtilities.fromScreenToSVGCoordinate(this.paper.canvas, x, y);
            return new draw2d.geo.Point(point.x, point.y);
        },

        /**
         * Get the best figure at a point on the canvas.
         *
         * The best figure is the one we should interact with.
         *
         * @param {number} x The point's x coordinate.
         * @param {number} y The point's y coordinate.
         * @param {draw2d.Figure} [draggedFigure] The figure being dragged.
         * @return {draw2d.Figure|null} The best figure at the given point or
         *     null if there are no figures to interact with.
         */
        getBestFigure: function (x, y, draggedFigure) {
            // Performance optimisation: only run the hit test if nothing,
            // a port, or a line resize handle is currently being dragged.
            var draggingPort = draggedFigure instanceof draw2d.Port,
                draggingLineResizeHandle = draggedFigure instanceof draw2d.shape.basic.LineResizeHandle;

            if (!draggedFigure || draggingPort || draggingLineResizeHandle) {
                return this._super(x, y, draggedFigure);
            }

            return null;
        },

        /**
         * Get the layer with the given name.
         *
         * @param {string} name The name of the layer to get.
         * @returns {JIRA.WorkflowDesigner.Draw2DCanvas.Layer|undefined} The requested layer or `undefined`.
         */
        getLayer: function (name) {
            return this._layers && this._layers[name];
        },

        /**
         * Gets the selected figure.
         *
         * @return {draw2d.Figure|null} The currently selected figure.
         */
        getSelectedFigure: function () {
            return this.getSelection().getPrimary();
        },

        /**
         * Calculate and return the canvas's current view box.
         *
         * @return {draw2d.geo.Rectangle} The canvas's current view box.
         */
        getViewBox: function () {
            var bottomRight,
                offset = jQuery(this.html).offset(),
                origin;

            // With no viewBox attribute, the origin is (0, 0).
            origin = this.paper.canvas.getAttribute("viewBox");
            origin = origin && _.map(origin.split(" "), parseFloat).slice(0, 2) || [0, 0];

            // Subtract the document's scroll because this method expects a point in screen space.
            bottomRight = this.fromDocumentToCanvasCoordinate(
                offset.left + this.svgElement.width() - jQuery(document).scrollLeft(),
                offset.top + this.svgElement.height() - jQuery(document).scrollTop()
            );

            return new draw2d.geo.Rectangle(
                origin[0],
                origin[1],
                bottomRight.x - origin[0],
                bottomRight.y - origin[1]
            );
        },

        /**
         * Get the current zoom level of the canvas.
         *
         * A value < 1.0 means that the canvas is zoomed out, > 1.0 is zoomed in.
         *
         * @return {number} The current zoom level of the canvas.
         */
        getZoom: function () {
            return this.svgElement.width() / this.getViewBox().getWidth();
        },

        /**
         * Remove a figure from the canvas.
         *
         * @param {draw2d.Figure} figure The figure.
         */
        removeFigure: function (figure) {
            this._super.apply(this, arguments);
            figure._layer && figure._layer.removeFigure(figure);
        },

        /**
         * Periodically resize the SVG element if it is present on the page.
         *
         * @private
         */
        _scheduleResize: function () {
            var defaultResizeInterval = JIRA.WorkflowDesigner.Draw2DCanvas.DEFAULT_RESIZE_INTERVAL,
                elementIsPresent = !!this.svgElement.closest("body").length,
                resizeInterval = _.isNumber(this._resizeInterval) ? this._resizeInterval : defaultResizeInterval;

            if (elementIsPresent) {
                setTimeout(this._scheduleResize, resizeInterval);
                this.fitToContainer();
            }
        },

        /**
         * Select the given figure.
         *
         * Pass null to deselect the current figure.
         *
         * @param {draw2d.Figure|null} figure The figure to select or null.
         */
        selectFigure: function (figure) {
            var canvas = this;

            this.editPolicy.each(function (index, editPolicy) {
                var isSingleSelectionPolicy = editPolicy instanceof draw2d.policy.canvas.SingleSelectionPolicy;
                if (isSingleSelectionPolicy) {
                    editPolicy.select(canvas, figure);
                }
            });
        },

        /**
         * Set how often the canvas resizes to fill its container.
         *
         * @param {number} resizeInterval The new resize interval (in milliseconds).
         */
        setResizeInterval: function (resizeInterval) {
            this._resizeInterval = resizeInterval;
        },

        /**
         * An optimised version of {@link draw2d.Canvas#setViewBox()}.
         *
         * This method is faster than the original but can result in minor,
         * temporary graphical glitches. Use this in almost all cases.
         *
         * @param {draw2d.geo.Rectangle} rectangle The new view box.
         */
        setViewBoxQuick: function (rectangle) {
            this.paper.canvas.setAttribute("viewBox", [
                rectangle.getX(),
                rectangle.getY(),
                rectangle.getWidth(),
                rectangle.getHeight()
            ].join(" "));

            // Call Raphael's (slow) implementation.
            this._debouncedSetViewBox(rectangle);
        },

        /**
         * Zoom in/out on a target point.
         *
         * @param {number} factor The zoom factor (e.g. 0.5 zooms in 2x, 2.0 zooms out 2x).
         * @param {draw2d.geo.Point} target The target point in canvas space.
         * @param {draw2d.geo.Rectangle} diagramBoundingBox Minimum bounding box containing all items on the canvas
         */
        zoom: function (factor, target, diagramBoundingBox) {
            this.setViewBoxQuick(JIRA.WorkflowDesigner.ViewBoxTransformer.getZoomViewBox({
                diagramBoundingBox: diagramBoundingBox,
                factor: factor,
                target: target,
                viewBox: this.getViewBox(),
                zoom: this.getZoom()
            }));
        }
    });

    /**
     * The default canvas resize interval (in milliseconds).
     *
     * @constant
     * @default
     * @type {number}
     */
    JIRA.WorkflowDesigner.Draw2DCanvas.DEFAULT_RESIZE_INTERVAL = 100;
}());
