(function () {
    AJS.namespace("JIRA.WorkflowDesigner.GlobalWorkflows");

    /**
     * @class
     * @classdesc Sets up the workflow designer on the global workflow page.
     */
    JIRA.WorkflowDesigner.GlobalWorkflows = function() {
        this._workflowData = this._getWorkflowData();
        this._workflowLinkContainer = jQuery("#workflow-links");
        this._workflowView = jQuery("#workflow-view-diagram");

        if (this._workflowView.length > 0) {
            this._setupPage();
        }
    };

    _.extend(JIRA.WorkflowDesigner.GlobalWorkflows.prototype,
    /** @lends JIRA.WorkflowDesigner.GlobalWorkflows.prototype */
    {
        /**
         * ID of the issue collector
         *
         * @constant
         * @type {string}
         * @default
         */
        ISSUE_COLLECTOR_ID: "9dd8e905",

        /**
         * Returns the meta-data of the current workflow.
         *
         * @return {{isDraft: boolean, isEditable: boolean, name: string, project: number}} workflow meta-data
         * @private
         */
        _getWorkflowData: function() {
            return {
                isDraft: !!jQuery(".status-draft").length,
                isEditable: !!jQuery("#edit-workflow-trigger").length,
                isInactive: !!jQuery(".status-inactive").length,
                name: jQuery(".workflow-name").text(),
                project: jQuery("#workflow-designer-project").val()
            };
        },

        /**
         * Decorates and configures the page.
         *
         * @private
         */
        _setupPage: function() {
            var doc = jQuery(document),
                instance = this;

            if (this._workflowData.isEditable) {
                this._canvas = this._workflowView.find("#workflow-canvas");
                this._initDesigner();
                this._workflowData.isDraft && this._triggerOpenAnalyticsEvent();
            } else {
                this._canvas = this._prepareViewPage();

                if (!this._workflowView.hasClass("hidden")) {
                    // Diagram tab is selected initially.
                    this._initDesigner();
                    this._triggerOpenViewModeAnalyticsEvent();
                } else {
                    // Diagram tab is not selected initially - postpone initialisation until it is selected.
                    // Container needs to be visible before we initialise the designer. It's made visible by another click handler.
                    // Delegate handling to document to make sure this is executed last.
                    doc.on("click", "#workflow-diagram", function(e) {
                        instance._initDesigner();
                        doc.off(e, "#workflow-diagram");
                    });
                }
            }

            this._createFeedbackButton();

            this._viewModeTriggers = this._workflowLinkContainer.find(".workflow-view-toggle");
            if (this._viewModeTriggers.length > 0) {
                this._configureViewPage();
            }
        },

        /**
         * Clears the current content of the diagram tab and adds a canvas to it.
         *
         * @private
         */
        _prepareViewPage: function() {
            this._workflowView.children().hide();
            return jQuery("<div id='workflow-canvas'>").appendTo(this._workflowView);
        },

        /**
         * Creates the workflow designer. Shows the unavailable message in IE8.
         *
         * @private
         */
        _initDesigner: function() {
            if (!JIRA.WorkflowDesigner.browserIsSupported()) {
                this._canvas.append(JIRA.WorkflowDesigner.Templates.GlobalWorkflows.notAvailableMessage());
                return;
            }

            new JIRA.WorkflowDesigner.Application({
                draft: this._workflowData.isDraft,
                element: this._canvas,
                immutable: !this._workflowData.isEditable,
                workflowId: this._workflowData.name
            });
        },

        /**
         * Triggers an analytic event for opening the workflow designer.
         * @private
         */
        _triggerOpenAnalyticsEvent: function() {
            AJS.EventQueue && AJS.EventQueue.push({
                name: "administration.workflow.open." +
                    (this._workflowData.isEditable ? "edit" : "view") + ".global"
            });
        },

        _triggerOpenViewModeAnalyticsEvent: function() {
            !this._workflowData.isInactive && this._triggerOpenAnalyticsEvent();
        },

        /**
         * Creates the feedback button and initialises Issue Collector.
         *
         * @private
         */
        _createFeedbackButton: function() {
            var instance = this,
                isBTF = !AJS.DarkFeatures.isEnabled('com.atlassian.jira.config.CoreFeatures.ON_DEMAND'),
                isSupportedBrowser = JIRA.WorkflowDesigner.browserIsSupported();

            if (isBTF || !isSupportedBrowser) {
                return;
            }

            this._feedbackButton = jQuery(JIRA.WorkflowDesigner.Templates.GlobalWorkflows.feedbackButton()).appendTo(this._workflowLinkContainer);

            JIRA.WorkflowDesigner.Feedback.init(this._feedbackButton.find("#workflow-designer-feedback-trigger"), this.ISSUE_COLLECTOR_ID).fail(function() {
                instance._feedbackButton.remove();
                instance._feedbackButton = null;
            });
        },

        /**
         * Configure the ViewPage
         * @private
         */
        _configureViewPage: function() {
            if (!this._viewModeTriggers.filter("#workflow-diagram").hasClass("active")) {
                this._feedbackButton && this._feedbackButton.hide();
            }

            this._viewModeTriggers.click(_.bind(this._onToggleViewMode, this));
        },

        /**
         * Handler for toggle view mode
         * @param {jQuery.Event} e
         * @private
         */
        _onToggleViewMode: function(e) {
            if (jQuery(e.target).attr("id") === "workflow-diagram") {
                this._feedbackButton && this._feedbackButton.show();
                this._triggerOpenViewModeAnalyticsEvent();
            } else {
                this._feedbackButton && this._feedbackButton.hide();
            }
        }
    });

    jQuery(function () {
        new JIRA.WorkflowDesigner.GlobalWorkflows();
    });
}());