(function () {
    var RectanglePatch = {
        /**
         * Calculate the union of two rectangles.
         *
         * @param {draw2d.geo.Rectangle} rectangle
         * @return {draw2d.geo.Rectangle} The union of <tt>rectangle</tt> and <tt>this</tt>.
         */
        union: function (rectangle) {
            var bottomRight,
                topLeft;

            bottomRight = {
                x: Math.max(rectangle.getRight(), this.getRight()),
                y: Math.max(rectangle.getBottom(), this.getBottom())
            };

            topLeft = {
                x: Math.min(rectangle.getLeft(), this.getLeft()),
                y: Math.min(rectangle.getTop(), this.getTop())
            };

            return new draw2d.geo.Rectangle(
                topLeft.x,
                topLeft.y,
                bottomRight.x - topLeft.x,
                bottomRight.y - topLeft.y
            );
        },

        /**
         * Creates a copy of this rectangle
         *
         * @returns {draw2d.geo.Rectangle} Cloned rectangle
         */
        clone: function () {
            return new draw2d.geo.Rectangle(
                this.x,
                this.y,
                this.w,
                this.h
            );
        },

        /**
         * Resize this Rectangle without changing the position of the center, using a scale factor.
         *
         * @param {number} factor Scale to apply to the rectangle (eg 1: no change, 2: twice as big)
         * @returns {draw2d.geo.Rectangle} The rectangle itself
         */
        scaleByFactor: function (factor) {
            return this.scale(this.w * (factor - 1), this.h * (factor - 1));
        },

        /**
         * Scale this rectangle's height by the value supplied as input and returns this for
         * convenience. This rectangle's height will become this.height + dh. The top left corner's
         * Y coordinate moves by -dh/2.
         *
         * @param {Number} dh  Amount by which height is to be re-sized
         * @return  {draw2d.geo.Rectangle} The rectangle itself
         **/
        scaleHeight: function (dh) {
            this.h += dh;
            this.y -= dh/2;
            this.adjustBoundary();
            return this;
        },

        /**
         * Scale this rectangle's width by the value supplied as input and returns this for
         * convenience. This rectangle's width will become this.width + dw. The top left corner's
         * X coordinate moves by -dw/2.
         *
         * @param {Number} dw  Amount by which width is to be re-sized
         * @return  {draw2d.geo.Rectangle} The rectangle itself
         **/
        scaleWidth: function (dw) {
            this.w += dw;
            this.x -= dw/2;
            this.adjustBoundary();
            return this;
        },

        /**
         * Gets the position of a Point relative to the position (top-left) corner of this
         * Rectangle
         * @param {draw2d.geo.Point} p Point to check
         * @returns {draw2d.geo.Point} Distance
         */
        getRelativePosition: function (p) {
            return p.subtract(this.getTopLeft());
        },

        /**
         * Gets the position of a Point relative to the position (top-left) corner of this
         * Rectangle as a percentage.
         *
         * @param {draw2d.geo.Point} p Point to check
         * @returns {draw2d.geo.Point} Point with percentage distances as components
         */
        getRelativePositionAsPercentage: function (p) {
            var position = this.getRelativePosition(p);
            return new draw2d.geo.Point(position.x / this.w, position.y / this.h);
        }
    };

    _.extend(draw2d.geo.Rectangle.prototype, RectanglePatch);
}());