(function (Backbone) {
    Backbone.define("JIRA.WorkflowDesigner.TransitionModel", Backbone.Model.extend(
    /** @lends JIRA.WorkflowDesigner.TransitionModel# */
    {
        defaults: {
            id: null,
            actionId: null,
            conditionsCount: null,
            description: null,
            globalTransition: false,
            name: null,
            postFunctionsCount: null,
            propertiesCount: null,
            screenId: null,
            screenName: null,
            source: null,
            sourceAngle: null,
            target: null,
            targetAngle: null,
            validatorsCount: null
        },

        /**
         * @classdesc
         * A transition to a target status.
         *
         * Global transitions don't have a source status, normal transitions do (and they may be the same).
         * @constructs
         * @extends Backbone.Model
         */
        initialize: function() {
            this._createId();

            this.on("change:actionId change:source change:target", _.bind(this._createId, this));
        },

        /**
         * Sets the unique ID of the transition.
         *
         * @private
         */
        _createId: function() {
            var actionId = this.getActionId(),
                prefix,
                source = this.get("source"),
                target = this.get("target");

            if (actionId && source && target) {
                prefix = source.isInitial() ? "IA" : "A";

                this.set("id", prefix + "<" + actionId + ":" + source.get("id") + ":" + target.get("id") + ">");
            }
        },

        /**
         * @returns {boolean} Whether the transition is global
         */
        isGlobalTransition: function () {
            return this.get("globalTransition");
        },

        /**
         * @returns {boolean} Whether the transition is the initial transition
         */
        isInitial: function () {
            return this.get("source").isInitial();
        },

        /**
         * @returns {string} The `actionId` for this transition
         */
        getActionId: function () {
            return this.get("actionId");
        },

        /**
         * @returns {number} The `stepId` of the transition's source
         */
        getSourceStepId: function () {
            var source = this.get("source");
            return source && source.get("stepId");
        },

        /**
         * @returns {number} The `stepId` of the transition's target
         */
        getTargetStepId: function () {
            var target = this.get("target");
            return target && target.get("stepId");
        },

        /**
         * Sets the source status and angle.
         * @param {JIRA.WorkflowDesigner.StatusModel} source New source status.
         * @param {number} targetAngle New source angle.
         */
        setSource: function(source, sourceAngle) {
            this.set({
                source: source,
                sourceAngle: sourceAngle
            });
        },

        /**
         * Sets the target status and angle.
         * @param {JIRA.WorkflowDesigner.StatusModel} target New target status.
         * @param {number} targetAngle New target angle.
         */
        setTarget: function(target, targetAngle) {
            this.set({
                target: target,
                targetAngle: targetAngle
            });
        },

        // We don't want to use Backbone's synchronisation logic.
        sync: jQuery.noop
    }));
}(JIRA.WorkflowDesigner.Backbone));