(function () {
    var messageDisabled;

    /**
     * @return {string} The workflow designer's "unsaved changes" message.
     */
    function getUnsavedChangesMessage() {
        return AJS.I18n.getText("workflow.designer.unsaved.changes");
    }

    AJS.namespace("JIRA.WorkflowDesigner.IO.LayoutAutoSaver");

    /**
     * Saves changes made in the workflow designer.
     *
     * Also shows a message on unload if there are unsaved changes.
     *
     * @class
     * @param {object} options
     * @param {object} [options.window=window] The object on which to set `onbeforeunload`.
     * @param {JIRA.WorkflowDesigner.WorkflowModel} options.workflowModel The application's `WorkflowModel`.
     */
    JIRA.WorkflowDesigner.IO.LayoutAutoSaver = function (options) {
        _.bindAll(this, "_save");

        this._save = _.debounce(this._save, 1000);
        this._window = options.window || window;
        this._workflowModel = options.workflowModel;
        this._workflowModel.on("layoutChanged", this._onLayoutChanged, this);
    };

    JIRA.WorkflowDesigner.IO.LayoutAutoSaver.prototype = {
        /**
         * Destroy the layout auto saver, removing its `onbeforeunload` handler.
         */
        destroy: function () {
            var shouldRestore = this._window.onbeforeunload === getUnsavedChangesMessage;

            if (shouldRestore) {
                this._window.onbeforeunload = this._originalOnBeforeUnload;
            }
        },

        /**
         * Install an unsaved changes message to be shown on unload (browse away, refresh).
         *
         * @private
         */
        _installUnsavedChangesMessage: function () {
            var shouldSet = !messageDisabled && this._window.onbeforeunload !== getUnsavedChangesMessage;

            if (shouldSet) {
                this._originalOnBeforeUnload = this._window.onbeforeunload;
                this._window.onbeforeunload = getUnsavedChangesMessage;
            }
        },

        /**
         * Handler for the "layoutChanged" event.
         *
         * @private
         */
        _onLayoutChanged: function () {
            this._installUnsavedChangesMessage();
            this._layoutId = _.uniqueId();
            this._save();
        },

        /**
         * Remove the 'Unsaved Changes' message.
         */
        removeUnsavedChangesMessage: function () {
            this._window.onbeforeunload = this._originalOnBeforeUnload;
            delete this._originalOnBeforeUnload;
        },

        /**
         * Save the current state of the workflow designer to the server.
         *
         * @private
         */
        _save: function () {
            var instance = this,
                layoutId = this._layoutId,
                request;

            function removeUnsavedChangedMessageForCurrentLayout() {
                if (instance._layoutId === layoutId) {
                    instance.removeUnsavedChangesMessage();
                }
            }

            request = JIRA.WorkflowDesigner.IO.AJAX.WorkflowAJAXManager.save(this._workflowModel);
            request.done(removeUnsavedChangedMessageForCurrentLayout);
            request.fail(function (errorMessage) {
                JIRA.WorkflowDesigner.Messages.showErrorMessage(errorMessage);
            });
        }
    };

    /**
     * Disables unsaved changes message.
     *
     * Needed to run JIRA WebDriver tests that fail when the message is shown.
     */
    JIRA.WorkflowDesigner.IO.LayoutAutoSaver.disableMessage = function() {
        messageDisabled = true;
    };
}());