(function (Backbone) {
    Backbone.define("JIRA.WorkflowDesigner.Canvas", Backbone.Marionette.ItemView.extend(
    /** @lends JIRA.WorkflowDesigner.Canvas# */
    {
        template: JIRA.WorkflowDesigner.Templates.container,

        ui: {
            container: ".container"
        },

        /**
         * Initialise the canvas.
         *
         * @classdesc The workflow designer's root view.
         * @constructs
         * @param {object} options
         * @param {boolean} [options.actions=true] Whether to show the actions bar.
         * @param {boolean} [options.fullScreenButton=true] Whether full screen mode is enabled.
         * @param {boolean} [options.immutable=false] Whether the workflow designer should be read-only.
         * @param {JIRA.WorkflowDesigner.WorkflowModel} options.workflowModel The application's workflow model.
         */
        initialize: function (options) {
            options = _.defaults(options, {
                actions: true,
                fullScreenButton: true,
                immutable: false
            });

            this._browserZoomChecker = new JIRA.WorkflowDesigner.BrowserZoomChecker();
            this._canvasModel = new JIRA.WorkflowDesigner.CanvasModel({}, {
                workflowModel: options.workflowModel
            });

            this._canvasView = new JIRA.WorkflowDesigner.CanvasView({
                canvasModel: this._canvasModel,
                immutable: options.immutable,
                workflowModel: options.workflowModel
            });

            options.actions && (this._actionsView = this._createActionsView(options));
            options.immutable || (this._propertiesPanelView = this._createPropertiesPanelView(options));
            this._zoomControlView = this._createZoomControlView();
        },

        /**
         * Auto fit the workflow to the canvas.
         */
        autoFit: function () {
            this._canvasView.autoFit();
        },

        /**
         * @param {object} options
         * @param {boolean} [options.fullScreenButton=true] Whether full screen mode is enabled.
         * @param {boolean} [options.immutable=false] Whether the workflow designer should be read-only.
         * @param {JIRA.WorkflowDesigner.WorkflowModel} options.workflowModel The application's `WorkflowModel`.
         * @private
         * @return {JIRA.WorkflowDesigner.ActionsView} An `ActionsView`.
         */
        _createActionsView: function (options) {
            var actionsView;

            actionsView = new JIRA.WorkflowDesigner.ActionsView({
                canvasModel: this._canvasModel,
                fullScreenButton: options.fullScreenButton,
                immutable: options.immutable,
                workflowModel: options.workflowModel
            });

            this.listenTo(actionsView, {
                "addStatus:done": this.hideProgressIndicator,
                "addStatus:submit": this.showProgressIndicator,
                "fullScreen:enter": this._enterFullScreen,
                "fullScreen:exit": this._exitFullScreen
            });

            return actionsView;
        },

        /**
         * @param {object} options
         * @param {JIRA.WorkflowDesigner.WorkflowModel} options.workflowModel The application's `WorkflowModel`.
         * @private
         * @return {JIRA.WorkflowDesigner.PropertiesPanel.PropertiesPanelView} A `PropertiesPanelView`.
         */
        _createPropertiesPanelView: function (options) {
            var propertiesPanelView;

            propertiesPanelView = new JIRA.WorkflowDesigner.PropertiesPanel.PropertiesPanelView({
                canvasModel: this._canvasModel,
                workflowModel: options.workflowModel
            });

            this.listenTo(propertiesPanelView, {
                "transaction:end": this.hideProgressIndicator,
                "transaction:start": this.showProgressIndicator
            });

            return propertiesPanelView;
        },

        /**
         * @private
         * @return {JIRA.WorkflowDesigner.ZoomControlView} A `ZoomControlView`.
         */
        _createZoomControlView: function () {
            var options, zoomControlView;

            options = {
                maximumZoomLevel: 2.0,
                minimumZoomLevel: 0.5,
                model: this._canvasModel
            };

            zoomControlView = new JIRA.WorkflowDesigner.ZoomControlView(options);

            this.listenTo(zoomControlView, {
                zoomIn: _.bind(this._canvasView.zoom, this._canvasView, 0.75),
                zoomOut: _.bind(this._canvasView.zoom, this._canvasView, 1.25)
            });

            return zoomControlView;
        },

        /**
         * Resize the workflow designer's canvas to occupy the whole window.
         *
         * @private
         */
        _enterFullScreen: function () {
            var canvasView = this._canvasView;

            // Decrease the resize interval to ensure smooth scaling.
            canvasView.setResizeInterval(25);

            JIRA.WorkflowDesigner.FullScreenController.enterFullScreen(this.el).done(function () {
                canvasView.setResizeInterval(JIRA.WorkflowDesigner.Draw2DCanvas.DEFAULT_RESIZE_INTERVAL);
            });
        },

        /**
         * Restore the workflow designer's canvas to its original position/size.
         *
         * @private
         */
        _exitFullScreen: function () {
            var canvasView = this._canvasView,
                instance = this;

            // Decrease the resize interval to ensure smooth scaling.
            canvasView.setResizeInterval(25);

            JIRA.WorkflowDesigner.FullScreenController.exitFullScreen(this.el).done(function () {
                canvasView.setResizeInterval(JIRA.WorkflowDesigner.Draw2DCanvas.DEFAULT_RESIZE_INTERVAL);
                instance.$el.css("width", "auto");
            });
        },

        /**
         * Hide the canvas's progress indicator (spinner).
         */
        hideProgressIndicator: function () {
            this._canvasView.setLocked(false);
            this._progressIndicator.hide();
        },

        /**
         * Render sub-views after the canvas has rendered.
         */
        onRender: function () {
            this.$el.addClass("workflow-designer");

            this._actionsView && this.$el.prepend(this._actionsView.render().el);
            this._canvasView.render(this.ui.container);
            this._propertiesPanelView && this.ui.container.append(this._propertiesPanelView.render().el);
            this.ui.container.append(this._zoomControlView.render().el);
        },

        /**
         * Remove the canvas from the page.
         */
        remove: function () {
            Backbone.Marionette.ItemView.prototype.remove.apply(this, arguments);
            this._browserZoomChecker.destroy();
            this._canvasView.remove();
        },

        /**
         * Show a progress indicator (spinner) in the canvas.
         */
        showProgressIndicator: function () {
            var options = {container: this.el},
                ProgressIndicatorBlanketView = JIRA.WorkflowDesigner.ProgressIndicatorBlanketView;

            this._canvasView.setLocked(true);
            this._progressIndicator || (this._progressIndicator = new ProgressIndicatorBlanketView(options));
            this._progressIndicator.show();
        }
    }));
}(JIRA.WorkflowDesigner.Backbone));