(function (Backbone) {
    /**
     * Show a "browser not supported" message in an element.
     *
     * @param {element} element The element to show the message in.
     */
    function showBrowserNotSupportedWarning(element) {
        var template = JIRA.WorkflowDesigner.Templates.browserNotSupportedWarning;
        jQuery(template()).appendTo(element);
    }

    AJS.namespace("JIRA.WorkflowDesigner.Application");

    /**
     * @class
     * @classdesc The JIRA workflow designer.
     * @constructor
     * @param {object} options
     * @param {boolean} [options.actions=true] Whether to show the actions bar.
     * @param {number} [options.currentStepId] The step ID of the status that is to be highlighted as "current".
     * @param {boolean} [options.draft=false] Whether the draft version of the workflow described by <tt>workflowId</tt> should be loaded.
     * @param {element} options.element The element to show the workflow designer in.
     * @param {boolean} [options.fullScreenButton=true] Whether full screen mode is enabled.
     * @param {boolean} [options.immutable=false] Whether the workflow designer should be read-only.
     * @param {object} [options.layoutData] The layout data to display.
     * @param {string} [options.workflowId] The ID of the workflow to load.
     */
    JIRA.WorkflowDesigner.Application = function (options) {
        options = _.defaults({}, options, {
            draft: false,
            immutable: false
        });

        if (!JIRA.WorkflowDesigner.browserIsSupported()) {
            showBrowserNotSupportedWarning(options.element);
            return;
        }

        this._workflowModel = new JIRA.WorkflowDesigner.WorkflowModel({
            currentStepId: options.currentStepId,
            draft: options.draft,
            name: options.workflowId
        });

        this._canvas = this._createCanvas(options).render();

        if (!options.immutable) {
            this._layoutAutoSaver = new JIRA.WorkflowDesigner.IO.LayoutAutoSaver({
                workflowModel: this._workflowModel
            });
        }

        this._initialiseWorkflow(options);
    };

    _.extend(JIRA.WorkflowDesigner.Application.prototype, Backbone.Events,
    /** @lends JIRA.WorkflowDesigner.Application# */
    {
        /**
         * @param {object} options Options to pass to the constructor.
         * @private
         * @return {JIRA.WorkflowDesigner.Canvas} A `JIRA.WorkflowDesigner.Canvas`.
         */
        _createCanvas: function (options) {
            options = _.extend(options, {
                el: options.element,
                workflowModel: this._workflowModel
            });

            return new JIRA.WorkflowDesigner.Canvas(options);
        },

        /**
         * Destroy the workflow designer.
         */
        destroy: function () {
            this._canvas && this._canvas.remove();
            this._layoutAutoSaver && this._layoutAutoSaver.destroy();
        },

        /**
         * Discard the draft of the workflow that is currently shown in the workflow designer.
         * When called, this function will disable the workflow designer until the operation has completed.
         * If a failure occurred, the workflow designer will be enabled again and a message will be shown to the user.
         *
         * @param {object} [options]
         * @param {boolean} [options.reloadDesigner=true] Whether the workflow designer should be reloaded with a new draft after the operation.
         * @returns {jQuery.Promise} A promise that is resolved on success.
         */
        discardDraft: function(options) {
            options = _.defaults({}, options, {
                reloadDesigner: true
            });
            this._canvas.showProgressIndicator();

            return JIRA.WorkflowDesigner.IO.AJAX.WorkflowAJAXManager.discard(this._workflowModel.get("name"))
                    .pipe(_.bind(this._onPublishOrDiscardSuccessHandler, this, options.reloadDesigner),
                            _.bind(this._onPublishOrDiscardFailHandler, this));
        },

        /**
         * Publish the draft of the workflow that is currently shown in the workflow designer.
         * When called, this function will disable the workflow designer until the operation has completed.
         * If a failure occurred, the workflow designer will be enabled again and a message will be shown to the user.
         *
         * @param {object} [options]
         * @param {boolean} [options.reloadDesigner=true] Whether the workflow designer should be reloaded with a new draft after the operation.
         * @returns {jQuery.Promise} A promise that is resolved on success.
         */
        publishDraft: function(options) {
            options = _.defaults({}, options, {
                reloadDesigner: true
            });
            this._canvas.showProgressIndicator();

            return JIRA.WorkflowDesigner.IO.AJAX.WorkflowAJAXManager.publish(this._workflowModel)
                    .pipe(_.bind(this._onPublishOrDiscardSuccessHandler, this, options.reloadDesigner),
                            _.bind(this._onPublishOrDiscardFailHandler, this));
        },

        /**
         * Initialise a workflow into the designer.
         * @param {object} options
         * @param {boolean} [options.draft=false] Whether the draft version of the workflow described by <tt>workflowId</tt> should be loaded.
         * @param {object} [options.layoutData] The workflow data to load.
         * @param {string} [options.workflowId] The ID of the workflow to load.
         * @private
         */
        _initialiseWorkflow: function (options) {
            var instance = this,
                request = this._loadWorkflow(options);

            request.done(function () {
                instance._workflowModel.set("loadedAt", new Date());
                instance._canvas.autoFit();
                instance.trigger("loaded");
            });
        },

        /**
         * Load a workflow into the designer.
         *
         * @param {object} options
         * @param {boolean} [options.draft=false] Whether the draft version of the workflow described by <tt>workflowId</tt> should be loaded.
         * @param {object} [options.layoutData] The workflow data to load.
         * @param {string} [options.workflowId] The ID of the workflow to load.
         * @returns {jQuery.Promise} A promise that is resolved on success.
         * @private
         */
        _loadWorkflow: function (options) {
            var deferred = jQuery.Deferred(),
                instance = this,
                request;

            if (options.layoutData) {
                request = jQuery.Deferred().resolve(options.layoutData);
            } else if (options.workflowId) {
                request = JIRA.WorkflowDesigner.IO.AJAX.WorkflowAJAXManager.load(options.workflowId, !!options.draft);
            }

            if (request) {
                this._canvas.showProgressIndicator();

                request.always(function () {
                    instance._canvas.hideProgressIndicator();
                });

                request.done(function (layoutData) {
                    instance._workflowModel.reset(layoutData);
                    deferred.resolve();
                });

                request.fail(function (errorMessage) {
                    JIRA.WorkflowDesigner.Messages.showErrorMessage(errorMessage);
                    deferred.reject(errorMessage);
                });
            } else {
                deferred.resolve();
            }

            return deferred.promise();
        },

        _onPublishOrDiscardSuccessHandler: function (reloadDesigner) {
            this._layoutAutoSaver.removeUnsavedChangesMessage();
            if (reloadDesigner) {
                return this._loadWorkflow({
                    workflowId: this._workflowModel.get("name"),
                    draft: this._workflowModel.get("draft")
                });
            }
        },

        _onPublishOrDiscardFailHandler: function (errorMessage) {
            JIRA.WorkflowDesigner.Messages.showErrorMessage(errorMessage);
            this._canvas.hideProgressIndicator();
        }
    });
}(JIRA.WorkflowDesigner.Backbone));