/**
 * Renders the status of the optimizer plugin (NOT_STARTED, IN_PROGRESS, DONE etc)
 * In future will handle things like displaying progress
 */
define('jira/optimizer-plugin/statusView', ['jira-optimizer-plugin/marionette-2.1.0', 'jira/moment'], function (Marionette, moment) {
    'use strict';

    var POLL_INTERVAL = 3000;

    return Marionette.ItemView.extend({
        template: function template(data) {
            if (data.lastScan) {
                // don't convert if 0 (used in soy template to display "never")
                data.lastScan = moment(data.lastScan).format("dddd, MMMM Do YYYY, h:mm:ss a");
            }
            if (data.values && data.values.length) {
                data.warningsCount = data.values.length;
            }
            return CustomfieldAnalyzer.Templates.Status(data);
        },

        ui: {
            scanButton: '#jgp-scan'
        },
        events: {
            'click @ui.scanButton': 'initializeScan'
        },
        onRender: function onRender() {
            // Display progressbar and start polling when initialized at IN_PROGRESS state
            if (this.model.get('status') === 'IN_PROGRESS') {
                setTimeout(this.initializeProgressTracking.bind(this), 0);
            }
        },
        fetchStatus: function fetchStatus() {
            var _this = this;

            return this.model.fetch().done(function (response) {
                if (_this.interval) {
                    // We are polling
                    switch (response.status) {
                        case 'IN_PROGRESS':
                            {
                                _this.updateProgress();
                                break;
                            }
                        case 'DONE':
                        case 'FAILED':
                        case 'STOPPED':
                        default:
                            clearInterval(_this.interval);
                            _this.updateProgress(); // Displays progressbar completion
                            _this.timeout = setTimeout(_this.render, 1000); // Delay for progress completion animation
                    }
                }
            }).fail(function () {
                //TBD handle this scenario - display message/flag etc
                clearInterval(_this.interval);
            });
        },
        initializeScan: function initializeScan() {
            var _this2 = this;

            this.ui.scanButton.prop('disabled', true);
            this.model.save().done(this.initializeProgressTracking.bind(this)).fail(function () {
                _this2.ui.scanButton.prop('disabled', false);
            });
        },
        initializeProgressTracking: function initializeProgressTracking() {
            // We do DOM manipulation instead of full re-render to keep progressbar animation
            this.$('.scan-status-text').addClass('hidden');
            this.$('.in-progress-text').removeClass('hidden');
            this.$('#recommendation-progress-indicator').removeClass('hidden');
            this.updateProgress();
            this.pollStatus();
        },
        pollStatus: function pollStatus() {
            this.interval = setInterval(this.fetchStatus.bind(this), POLL_INTERVAL);
        },
        updateProgress: function updateProgress() {
            AJS.progressBars.update("#recommendation-progress-indicator", this.model.get('progress'));
        },
        destroy: function destroy() {
            this.interval && clearInterval(this.interval);
            this.timeout && clearTimeout(this.timeout);
            Marionette.ItemView.prototype.destroy.call(this);
        }
    });
});