package com.atlassian.jira.plugins.importer.github.importer;

import com.atlassian.jira.config.ConstantsManager;
import com.atlassian.jira.issue.issuetype.IssueType;
import com.atlassian.jira.issue.resolution.Resolution;
import com.atlassian.jira.plugins.importer.external.beans.ExternalAttachment;
import com.atlassian.jira.plugins.importer.external.beans.ExternalComment;
import com.atlassian.jira.plugins.importer.external.beans.ExternalIssue;
import com.atlassian.jira.plugins.importer.external.beans.ExternalProject;
import com.atlassian.jira.plugins.importer.github.config.ConfigBean;
import com.atlassian.jira.plugins.importer.github.config.SchemeStatusMapping;
import com.atlassian.jira.plugins.importer.github.fetch.GithubConstants;
import com.atlassian.jira.plugins.importer.github.fetch.Project;
import com.atlassian.jira.plugins.importer.github.importer.markup.ConversionResult;
import com.atlassian.jira.plugins.importer.github.importer.markup.MarkupConverter;
import com.atlassian.jira.plugins.importer.github.rest.workflow.WorkflowService;
import com.atlassian.jira.plugins.importer.imports.importer.ImportLogger;
import com.atlassian.jira.util.I18nHelper;
import com.atlassian.jira.workflow.JiraWorkflow;
import com.atlassian.jira.workflow.WorkflowManager;
import com.atlassian.jira.workflow.WorkflowSchemeManager;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.base.Predicates;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableListMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import com.google.common.collect.Sets;
import org.eclipse.egit.github.core.Comment;
import org.eclipse.egit.github.core.Issue;
import org.eclipse.egit.github.core.Label;

import javax.annotation.Nullable;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

public class GithubIssueConverter {


    private final ConfigBean configBean;
    private final ConstantsManager constantsManager;
    private final WorkflowService workflowService;
    private final AttachmentManager attachmentManager = new AttachmentManager();
	private final Multimap<ExternalIssue, Attachment> issueAttachments = HashMultimap.create();
	private final MarkupConverter converter;
	private final I18nHelper i18nHelper;

    public GithubIssueConverter(ConfigBean configBean, ConstantsManager constantsManager, WorkflowManager workflowManager, WorkflowSchemeManager workflowSchemeManager,
								DataBean dataBean, final I18nHelper i18nHelper) {
        this.configBean = configBean;
        this.constantsManager = constantsManager;
		this.i18nHelper = i18nHelper;
		workflowService = new WorkflowService(workflowManager, workflowSchemeManager, constantsManager);
		this.converter = new MarkupConverter(dataBean, configBean);
    }

    public ExternalIssue convertIssue(final Project project, final Issue issue, final ImportLogger importLogger, ExternalProject externalProject) {
        final ExternalIssue externalIssue = new ExternalIssue();
        externalIssue.setExternalId(String.valueOf(issue.getNumber()));
        externalIssue.setReporter(issue.getUser().getLogin());
        if( issue.getAssignee() != null ) {
            externalIssue.setAssignee(issue.getAssignee().getLogin());
        }
        externalIssue.setSummary(issue.getTitle());
        externalIssue.setCreated(issue.getCreatedAt());
        externalIssue.setUpdated(issue.getUpdatedAt());

        String description = issue.getBody();
        if( description != null ) {
			final ConversionResult convert = converter.convert(project.getRepository(), description);
			issueAttachments.putAll(externalIssue, convert.getAttachments());
			description = convert.getConvertedContent();
		}
        externalIssue.setDescription(description);

        IssueType issueType = applyLabels(issue, importLogger, externalIssue);
		if(issueType == null) {
			issueType = configBean.getDefaultIssueType(externalProject.getKey());
			externalIssue.setIssueType(issueType.getName());
		}

        // apply status (after type is known)
        JiraWorkflow workflow = workflowService.getWorkflow(configBean.getSchemeStatusMapping().getWorkflowSchemeName(), issueType.getId());
        SchemeStatusMapping.JiraStatusMapping jiraStatusMapping = configBean.getSchemeStatusMapping().getWorkflowIdToStatusMapping().get(workflow.getName());

        if( issue.getState().equals(GithubConstants.GITHUB_ISSUE_STATE_OPEN) ) {
            externalIssue.setStatus( jiraStatusMapping.getOpenStatus() );
        } else if( issue.getState().equals(GithubConstants.GITHUB_ISSUE_STATE_CLOSED) ) {
            externalIssue.setStatus( jiraStatusMapping.getClosedStatus() );
        } else {
            throw new IllegalStateException("Unknown GitHub issue state: "+issue.getState());
        }

        // comments
        List<Comment> comments = configBean.getGithubDataService().getComments(issue);
        externalIssue.setComments(Lists.newArrayList(Iterables.transform(comments, new Function<Comment, ExternalComment>() {
            public ExternalComment apply(@Nullable Comment comment) {
                String body = comment.getBody();
                if( body != null ) {
					final ConversionResult convert = converter.convert(project.getRepository(), body);
					issueAttachments.putAll(externalIssue, convert.getAttachments());
					body = convert.getConvertedContent();
                }
                return new ExternalComment(body, comment.getUser().getLogin(), comment.getCreatedAt());
            }
        })));

        // version
        if( issue.getMilestone() != null ) {
            externalIssue.setFixedVersions(Lists.newArrayList(issue.getMilestone().getTitle()));
        }

        return externalIssue;
    }

	@VisibleForTesting
	MappingResult getMapping(Iterable<Label> labels, final Map<String, Integer> mapping) {
		final HashSet<Integer> validResults = Sets.newHashSet(Iterables.filter(Iterables.transform(labels, new Function<Label, Integer>() {
			@Override
			public Integer apply(Label input) {
				return mapping.get(input.getName());
			}
		}), Predicates.notNull()));
		final ImmutableListMultimap<Integer,Label> resultToLabels = Multimaps.index(labels, new Function<Label, Integer>() {
			@Override
			public Integer apply(Label input) {
				return mapping.containsKey(input.getName()) ? mapping.get(input.getName()) : -1;
			}
		});

		if(validResults.size() == 1) {
			final Integer validId = validResults.iterator().next();
			return MappingResult.found(validId, resultToLabels.get(validId));
		} else if(validResults.size() == 0) {
			return MappingResult.noResult();
		} else { //more than 1!
			final Iterable<Label> conflictingLabels = Iterables.concat(Iterables.transform(validResults, new Function<Integer, Iterable<Label>>() {
				@Override
				public Iterable<Label> apply(Integer input) {
					return resultToLabels.get(input);
				}
			}));
			return MappingResult.conflict(conflictingLabels);
		}
	}

    /**
     * Apply issue type, resolution and labels. For convenience returns the complete {@link IssueType}.
     */
	@VisibleForTesting
    IssueType applyLabels(Issue issue, ImportLogger importLogger, ExternalIssue externalIssue) {

		final MappingResult issueTypeResult = getMapping(issue.getLabels(), configBean.getIssueTypeMapping());
		final MappingResult resolutionResult = getMapping(issue.getLabels(), configBean.getResolutionMappings());
		final HashSet<Label> usedLabels = Sets.newHashSet();

	    IssueType issueType = null;
		if(issueTypeResult.hasMapping()) {
			issueType = constantsManager.getIssueTypeObject(String.valueOf(issueTypeResult.resultId));
			externalIssue.setIssueType(issueType.getName());
			usedLabels.addAll(Lists.newArrayList(issueTypeResult.sourceLabels));
		} else if(issueTypeResult.isConflict()) {
			importLogger.warn(i18nHelper.getText("com.atlassian.jira.plugins.importer.github.import.conflict.issue.types",
					"" + issue.getNumber(), Joiner.on(", ").join(resolutionResult.sourceLabels)));
		}

		if (resolutionResult.hasMapping()) {
			final Resolution resolution = constantsManager.getResolutionObject(String.valueOf(resolutionResult.resultId));
			externalIssue.setResolution(resolution.getName());
			usedLabels.addAll(Lists.newArrayList(resolutionResult.sourceLabels));
		} else if(resolutionResult.isConflict()) {
			importLogger.warn(i18nHelper.getText("com.atlassian.jira.plugins.importer.github.import.conflict.resolutions",
					"" + issue.getNumber(), Joiner.on(", ").join(resolutionResult.sourceLabels)));
		}

		if(configBean.isAutoLabels()) {
			final HashSet<Label> unusedLabels = Sets.newHashSet(issue.getLabels());
			Iterables.removeAll(unusedLabels, usedLabels);
			final Iterable<String> labelNames = Iterables.transform(unusedLabels, new Function<Label, String>() {
				@Override
				public String apply(Label input) {
					return input.getName();
				}
			});
			externalIssue.setLabels(ImmutableList.copyOf(labelNames));
		}
        return issueType;
    }

    public List<ExternalAttachment> downloadAttachmentsForIssue(ExternalIssue externalIssue, ImportLogger importLogger) {
        return attachmentManager.downloadAttachmentsForIssue(issueAttachments.get(externalIssue), importLogger);
    }

	@VisibleForTesting
	static class MappingResult {
		private static final int CONFLICT = -2;
		private int resultId = -1;
		private Iterable<Label> sourceLabels;

		public Iterable<Label> getSourceLabels() {
			return sourceLabels;
		}

		public int getResultId() {
			return resultId;
		}

		public boolean hasMapping() {
			return resultId > 0;
		}

		public boolean isConflict() {
			return resultId == CONFLICT;
		}

		private MappingResult() {}

		private MappingResult(int resultId, Iterable<Label> sourceLabels) {
			this.resultId = resultId;
			this.sourceLabels = sourceLabels;
		}

		public static MappingResult noResult() {
			return new MappingResult();
		}

		public static MappingResult found(int result, Iterable<Label> labels) {
			return new MappingResult(result, labels);
		}

		public static MappingResult conflict(Iterable<Label> labels) {
			return new MappingResult(CONFLICT, labels);
		}

	}
}
