package com.atlassian.jira.plugins.importer.github;

import com.atlassian.jira.plugins.importer.github.config.ConfigBean;
import com.atlassian.jira.plugins.importer.github.fetch.DataFetchJob;
import com.atlassian.jira.plugins.importer.github.fetch.GithubDataService;
import com.atlassian.jira.plugins.importer.github.fetch.auth.GithubAuthenticator;
import com.atlassian.jira.plugins.importer.github.fetch.auth.GithubAuthenticatorImpl;
import com.atlassian.jira.plugins.importer.github.importer.DataBean;
import com.atlassian.jira.plugins.importer.github.web.GithubAuthenticationPage;
import com.atlassian.jira.plugins.importer.github.web.GithubFetchDataPage;
import com.atlassian.jira.plugins.importer.github.web.GithubProjectMappingPage;
import com.atlassian.jira.plugins.importer.github.web.GithubSetupPage;
import com.atlassian.jira.plugins.importer.github.web.GithubWorkflowPage;
import com.atlassian.jira.plugins.importer.github.web.GithubLabelMappingPage;
import com.atlassian.jira.plugins.importer.imports.importer.ImportDataBean;
import com.atlassian.jira.plugins.importer.imports.importer.JiraDataImporter;
import com.atlassian.jira.plugins.importer.web.AbstractImporterController;
import com.atlassian.jira.plugins.importer.web.AbstractSetupPage;
import com.atlassian.jira.plugins.importer.web.ConfigFileHandler;
import com.atlassian.jira.util.I18nHelper;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.sal.api.ApplicationProperties;
import com.google.common.collect.Lists;
import org.apache.commons.lang.StringUtils;

import java.util.List;

/**
 * Controls the GitHub import.
 */
public class GithubImporterController extends AbstractImporterController {

    public static final String IMPORT_CONFIG_BEAN = "com.atlassian.jira.plugins.importer.github.config";
    public static final String IMPORT_ID = "Github";
    public static final String EXTERNAL_SYSTEM = "com.atlassian.jira.plugins.jira-importers-github-plugin:githubImporterKey";

    private final ConfigFileHandler configFileHandler;
	private final I18nHelper i18nHelper;

    public GithubImporterController(@ComponentImport JiraDataImporter importer,
									@ComponentImport ConfigFileHandler configFileHandler,
									@ComponentImport I18nHelper i18nHelper) {
        super(importer, IMPORT_CONFIG_BEAN, IMPORT_ID);
        this.configFileHandler = configFileHandler;
		this.i18nHelper = i18nHelper;
	}

    @Override
    public boolean createImportProcessBean(AbstractSetupPage abstractSetupPage) {
        final GithubDataService githubDataService = new GithubDataService();
        final DataFetchJob dataFetchJob = new DataFetchJob(githubDataService);
        final ConfigBean configBean = new ConfigBean(githubDataService);
		final GithubAuthenticatorImpl githubAuthenticator = new GithubAuthenticatorImpl(i18nHelper);
		final GithubImportProcessBean bean = new GithubImportProcessBean(githubDataService, dataFetchJob, configBean, githubAuthenticator);
		final GithubSetupPage githubSetupPage = (GithubSetupPage) abstractSetupPage;

		if (!configFileHandler.populateFromConfigFile(githubSetupPage, configBean)) {
			return false;
		}

		if (!githubSetupPage.isGithubClassic()) {
			configBean.setGithubUrl(githubSetupPage.getGithubUrl());
			configBean.setGithubEnterprise(true);
			githubAuthenticator.setGithubEnterpriseUrl(githubSetupPage.getGithubUrl());
		}

		if(!validateGithubEnterprise(githubSetupPage, bean)) {
			return false;
		}

		storeImportProcessBeanInSession(bean);
        return true;
    }

	private boolean validateGithubEnterprise(GithubSetupPage githubSetupPage, GithubImportProcessBean processBean) {
		final GithubAuthenticator authenticator = processBean.getAuthenticator();

		final GithubAuthenticator.AuthorizationResponse authorizationResponse;
		if (StringUtils.isNotBlank(githubSetupPage.getGithubToken())) {
			authorizationResponse = authenticator.authenticateAndValidateGithubToken(githubSetupPage.getGithubToken());
		} else {
			authorizationResponse = authenticator.authenticateGithub(githubSetupPage.getGithubUsername(), githubSetupPage.getGithubPassword());
		}

		if(authorizationResponse.getState() == GithubAuthenticator.AuthState.ERROR) {
			githubSetupPage.addErrorMessage(authorizationResponse.getError());
			return false;
		} else if (authorizationResponse.getState() == GithubAuthenticator.AuthState.OK) {
			processBean.getGithubDataService().setGitHubClient(authenticator.buildAuthenticatedClient());
		}
		//in case of Two-factor authentication also return true
		return true;
	}

    @Override
    public ImportDataBean createDataBean() throws Exception {
        GithubImportProcessBean importProcessBean = (GithubImportProcessBean) getImportProcessBeanFromSession();
        if( importProcessBean == null ) {
            throw new IllegalStateException("No active importer");
        }

        ConfigBean configBean = (ConfigBean) importProcessBean.getConfigBean();
        return new DataBean(configBean, importProcessBean, i18nHelper);
    }

    // @Override no override annotation to remain compatible with JIRA 5
    public List<String> getStepNameKeys() {
        return Lists.newArrayList(
				"com.atlassian.jira.plugins.importer.github.step.setup",
                "com.atlassian.jira.plugins.importer.github.step.oauth",
                "com.atlassian.jira.plugins.importer.github.step.projectMapping",
                "com.atlassian.jira.plugins.importer.github.step.fetchData",
                "com.atlassian.jira.plugins.importer.github.step.labelMapping",
                "com.atlassian.jira.plugins.importer.github.step.workflow"
        );
    }

    @Override
    public List<String> getSteps() {
        return Lists.newArrayList(
				GithubSetupPage.class.getSimpleName(),
                GithubAuthenticationPage.class.getSimpleName(),
                GithubProjectMappingPage.class.getSimpleName(),
                GithubFetchDataPage.class.getSimpleName(),
                GithubLabelMappingPage.class.getSimpleName(),
                GithubWorkflowPage.class.getSimpleName()
        );
    }

}
