AJS.test.require([
    'com.atlassian.jira.plugins.jira-editor-plugin:round-trip',
], function () {

    var $ = require('jquery');
    var _ = require('underscore');
    var Analytics = require("jira/editor/analytics");

    module("RoundTrip", {
        setup: function () {
            this.clock = sinon.useFakeTimers();
            this.sandbox = sinon.sandbox.create();

            this.mockStorage = {
                getItem: function(key) {
                    return this[key];
                },
                setItem: function(key, value) {
                    this[key] = value;
                }
            };

            this.context = AJS.test.mockableModuleContext();
            this.context.mock('jira/data/local-storage', this.mockStorage);
            this.context.mock("jira/editor/analytics", Analytics);
            this.sandbox.stub(Analytics, "sendEvent");
            this.controller = this.context.mock("jira/editor/controller", {
                renderMarkup: function(markup) {
                    return {
                        "*wrong_": new $.Deferred().resolve("<p>*wrong_</p>"),
                        "*strong*": new $.Deferred().resolve("<p><b>strong</b></p>") }[markup] || new $.Deferred().reject()
                }
            });
        },

        teardown: function () {
            this.sandbox.restore();
            this.clock.restore();
        }
    });


    test('Should not send on first run', function () {
        var RoundTrip = this.context.require("jira/editor/round-trip");

        RoundTrip.verify("<p><b>strong</b></p>", "*strong*", {});

        this.clock.tick(30 * 1000);

        // should be ready after 30 seconds
        var score = RoundTrip.get();
        equal(score.average, 1);
        equal(score.count, 1);
        equal(Analytics.sendEvent.callCount, 0);
    });

    test('Should reset after wrap limit', function () {
        var RoundTrip = this.context.require("jira/editor/round-trip");

        for(var i = 0; i < 50; i++) {
            RoundTrip.verify("<p><b>strong</b></p>", "*strong*", {});
            this.clock.tick(30 * 1000);
        }

        var score = RoundTrip.get();
        equal(score.average, 1);
        equal(score.count, 50);


        for(var i = 0; i < 50; i++) {
            RoundTrip.verify("<p><b>wrong</b></p>", "*wrong_", {});
            this.clock.tick(30 * 1000);
        }

        // should be ready after 30 seconds
        var score = RoundTrip.get();
        equal(Math.floor(score.average * 100), 49);
        equal(score.count, 100);
        equal(Analytics.sendEvent.callCount, 50);
        equal(Analytics.sendEvent.getCalls().every(function(call) {
            return call.args[0] === "editor.instance.roundtrip.score.raw"
        }), true, "Only raw events");

        this.clock.tick(25 * 60 * 60 * 1000); // also trigger sendEvent
        RoundTrip.verify("<p><b>wrong</b></p>", "*wrong_", {});
        this.clock.tick(30 * 1000); // also trigger sendEvent

        score = RoundTrip.get();
        equal(Math.floor(score.average * 100), 0);
        equal(score.count, 1);
        equal(Analytics.sendEvent.callCount, 52);
        deepEqual(_.countBy(Analytics.sendEvent.getCalls(), function(call) {
            return call.args[0];
        }), { "editor.instance.roundtrip.score.raw": 51, "editor.instance.roundtrip.score": 1 });
    });

    test('Should not send event when localStorage is full', function () {
        var RoundTrip = this.context.require("jira/editor/round-trip");

        for(var i = 0; i < 50; i++) {
            RoundTrip.verify("<p><b>strong</b></p>", "*strong*", {});
            this.clock.tick(30 * 1000);
        }

        var score = RoundTrip.get();
        equal(score.average, 1);
        equal(score.count, 50);
        equal(Analytics.sendEvent.callCount, 0);

        this.mockStorage.setItem = this.sandbox.stub()["throws"]("QUOTA_EXCEEDED_ERR");

        this.clock.tick(25 * 60 * 60 * 1000); // also trigger sendEvent
        RoundTrip.verify("*wrong_", {});
        this.clock.tick(30 * 1000); // also trigger sendEvent

        equal(Analytics.sendEvent.callCount, 0);
    });

    test('Should normalize html', function () {
        var RoundTrip = this.context.require("jira/editor/round-trip");

        this.sandbox.stub(this.context.require("jira/editor/controller"), "renderMarkup")
            .returns(new $.Deferred().resolve("<p><strong>strong</strong></p><p>text</p>", "*strong*\n\ntext"));
        RoundTrip.verify("<p><b>strong</b></p>\n\n<p>text</p>", "*strong*\n\ntext", {});
        this.clock.tick(30 * 1000);

        var score = RoundTrip.get();
        equal(score.average, 1);
        equal(Analytics.sendEvent.callCount, 0);
    });

    test('Verify scoring', function () {
        var RoundTrip = this.context.require("jira/editor/round-trip");
        var renderMarkup = this.sandbox.stub(this.context.require("jira/editor/controller"), "renderMarkup");

        var verifyRoundTrip = function (source, markup, target, expectedAverage) {
            this.mockStorage["jira.editor.score"] = null;
            renderMarkup.returns(new $.Deferred().resolve(target));
            RoundTrip.verify(source, markup, {});
            this.clock.tick(30 * 1000);

            var score = RoundTrip.get();
            equal(score.average, expectedAverage, source + " -> " + markup + " -> " + target + ' (' + expectedAverage + ')');
        }.bind(this);

        verifyRoundTrip("just text", "just text", "just text", 1);
        verifyRoundTrip("other text", "other text", "just text", 0);
        verifyRoundTrip("<p>unwrapped para</p>", "unwrapped para", "unwrapped para", 1);
        verifyRoundTrip("<b>bold</b>", "*bold*", "<b>bold</b>", 1);
        verifyRoundTrip("<strong>strong</strong>", "*strong*", "<b>bold</b>", 0.5); // 0.5 due to normalization
        verifyRoundTrip("<strong>strong</strong>", "*strong*", "<em>strong</em>", 0.5);
        verifyRoundTrip('this is <img src="data:base64" data-filename="image.png"/>',
            "this is !image.png!",
            'this is <span class="error">No usable issue stored in the context, unable to resolve filename \'image.png\'</span>', 1);
        verifyRoundTrip("just text <ul><li>nested <code>content</code></li></ul>",
            "just text\n- {{content}}",
            "just text <ul><li>nested <code>nonsense</code></li></ul>", 1 - 1/6);
        verifyRoundTrip('<div class="table-wrap"><table><td>cell</td></table></div>',
            "|cell|",
            "<table><td>cell</td></table>", 1);
        verifyRoundTrip('order of attr matters: <img src="image.png" height="119" width="111" style="border: 0px solid black;" data-mce-src="/jira/secure/attachment/10000/10000_image-2016-09-12-17-40-52-522.png" data-mce-style="border: 0px solid black;">',
            "order of attr matters: !image.png|width=111,height=119!",
            'order of attr matters: <img src="image.png" width="111" style="border: 0px solid black" height="119">', 1);
        verifyRoundTrip('<div class="code panel" style="border-width: 1px;"><div class="codeContent panelContent"><pre class="code-java">dupa</pre></div></div>',
                '{code:java}dupa{code}',
                '<div class="code panel" style="border-width: 1px;" data-mce-style="border-width: 1px;"><div class="codeContent panelContent"><pre class="code-java">dupa</pre></div></div>', 1);
    });
});