AJS.test.require([
    'com.atlassian.jira.plugins.jira-editor-plugin:resources',
    'com.atlassian.jira.plugins.jira-editor-plugin:api',
    'com.atlassian.jira.plugins.jira-editor-plugin:tinymce',
    'com.atlassian.jira.plugins.jira-editor-plugin:converter',
    'com.atlassian.jira.plugins.jira-editor-plugin:toggle',
    'com.atlassian.jira.plugins.jira-dnd-attachment-plugin:shared-utility'
], function () {
    "use strict";
    var $ = require("jquery");
    var _ = require("underscore");

    module('PastePlugin', {
        setup: function () {
            this.context = AJS.test.mockableModuleContext();
            this.sandbox = sinon.sandbox.create();
            sinon.useFakeXMLHttpRequest();

            this.mockModules();

            this.PastePlugin = this.context.require('jira/editor/plugins/paste');
        },

        mockModules: function () {
            this.sandbox.stub(JIRA.Editor.Tags.Templates, "image");

            this.jQueryMethods = {
                is: function () {
                },
                hasClass: function () {
                },
                attr: function () {
                },
                find: function () {
                },
                match: function () {
                },
                replaceWith: function () {
                },
                css: function () {
                },
                on: function () {
                },
                off: function () {
                },
                remove: function () {
                },
                length: 1
            };
            this.sandbox.stub(this.jQueryMethods, 'attr').returns(this.jQueryMethods);
            this.sandbox.stub(this.jQueryMethods, 'find').returns(this.jQueryMethods);
            this.sandbox.stub(this.jQueryMethods, 'is').returns(this.jQueryMethods);
            this.sandbox.stub(this.jQueryMethods, 'hasClass').returns(false);
            this.sandbox.stub(this.jQueryMethods, 'match').returns(this.jQueryMethods);
            this.sandbox.stub(this.jQueryMethods, 'replaceWith').returns(this.jQueryMethods);
            this.sandbox.stub(this.jQueryMethods, 'css').returns(this.jQueryMethods);
            this.sandbox.stub(this.jQueryMethods, 'on').returns(this.jQueryMethods);
            this.sandbox.stub(this.jQueryMethods, 'off').returns(this.jQueryMethods);
            this.sandbox.stub(this.jQueryMethods, 'remove').returns(this.jQueryMethods);

            this.$ = sinon.stub().returns(this.jQueryMethods);
            this.$.Event = sinon.stub().returns(this.jQueryMethods);
            this.$.Deferred = $.Deferred;
            this.context.mock('jquery', this.$);

            var params = {
                allowedAttachments: true
            };
            this.wrmData = {
                claim: sinon.stub()
            };
            this.context.mock("wrm/data", this.wrmData);
            this.wrmData.claim.withArgs('com.atlassian.jira.plugins.jira-editor-plugin:wrm.editor-data').returns(params);

            this.pasteUtility = {
                convertBlobToImage: sinon.stub(),
                dropFileToElement: sinon.stub(),
                generateFileName: sinon.stub()
            };
            this.context.mock('dndattachment/ctrlv/utility', this.pasteUtility);

            this.Strings = {
                startsWith: sinon.stub(),
                getFilenameFromHref: sinon.stub(),
                isAttachmentPath: sinon.stub(),
                isThumbnailPath: sinon.stub()
            };
            this.context.mock('jira/editor/converter/util/strings', this.Strings);

            this.EditorSchema = {
                sanitizeHtml: sinon.stub()
            };
            this.context.mock('jira/editor/schema', this.EditorSchema);

            this.editorInstance = {
                element: {
                    $wikiTextarea: {
                        trigger: sinon.stub()
                    }
                },
                contextManager: {
                    getPasteSchemaForContext: function () {
                        return this.schema
                    }.bind(this)
                }
            };

            this.Analytics = {
                sendEvent: sinon.stub()
            };
            this.context.mock('jira/editor/analytics', this.Analytics);
        },

        teardown: function () {
            this.sandbox.restore();
        }
    });

    test('Should sanitizeHtml', function () {
        this.PastePlugin.configure(this.editorInstance, {
            plugins: {
                push: sinon.stub()
            }
        }).paste_preprocess(null, {
            content: ''
        });

        sinon.assert.calledOnce(this.EditorSchema.sanitizeHtml);
    });

    test('Paste image from external resource', function () {
        this.$().is.withArgs('img').returns(true);
        var mock = _.extend({}, this.jQueryMethods, {0: {src: ""}});
        this.$().find.withArgs('img:first-child:last-child').returns(mock);
        this.$().match.withArgs(/^((data|blob):http)|(data:image)/).returns(false);

        this.PastePlugin.configure(this.editorInstance, {
            plugins: {
                push: sinon.stub()
            }
        }).paste_postprocess(null, {
            content: ''
        });

        sinon.assert.calledOnce(JIRA.Editor.Tags.Templates.image);
    });

    test('Paste image from clipboard', function () {
        this.$().is.withArgs('img').returns(true);
        var mock = _.extend({}, this.jQueryMethods, {0: {src: "blob:"}});
        this.$().find.withArgs('img:first-child:last-child').returns(mock);
        this.$().match.withArgs(/^((data|blob):http)|(data:image)/).returns(true);
        this.$().on.withArgs('dropHandled.dropFileToElement').returns(true);

        this.PastePlugin.configure(this.editorInstance, {
            plugins: {
                push: sinon.stub()
            }
        }).paste_postprocess(null, {
            content: ''
        });

        sinon.assert.calledThrice(this.Analytics.sendEvent);
        sinon.assert.calledWith(this.Analytics.sendEvent, "editor.instance.paste");
        sinon.assert.calledWith(this.Analytics.sendEvent, "editor.instance.paste.image");
        sinon.assert.calledWith(this.Analytics.sendEvent, "editor.instance.paste.addimage.nocache.http");
        sinon.assert.calledOnce(JIRA.Editor.Tags.Templates.image);
    });

    test('Paste image from clipboard with tinymce cache', function () {
        var blob = {};
        var cachedBlob = {blob: function () {
            return blob;
        }};
        this.editorInstance.editor = {
            editorUpload: {
                blobCache: {
                    getByUri: sinon.stub().returns(cachedBlob)
                }
            }
        };
        var $img = $('<img src="data:http:cachedUrl" />');
        this.$().is.withArgs('img').returns(true);
        this.$().find.withArgs('img:first-child:last-child').returns($img);
        this.$().match.withArgs(/^((data|blob):http)|(data:image)/).returns(true);
        this.$().on.withArgs('dropHandled.dropFileToElement').returns(true);
        this.pasteUtility.generateFileName.returns('generatedFilename');
        this.pasteUtility.dropFileToElement.returns(true);

        this.PastePlugin.configure(this.editorInstance, {
            plugins: {
                push: sinon.stub()
            }
        }).paste_postprocess(null, {
            content: ''
        });

        sinon.assert.calledOnce(JIRA.Editor.Tags.Templates.image);
        sinon.assert.calledOnce(this.pasteUtility.convertBlobToImage);
        sinon.assert.calledWith(this.pasteUtility.convertBlobToImage, blob, "generatedFilename.png");
        sinon.assert.calledThrice(this.Analytics.sendEvent);
        sinon.assert.calledWith(this.Analytics.sendEvent, "editor.instance.paste");
        sinon.assert.calledWith(this.Analytics.sendEvent, "editor.instance.paste.image");
        sinon.assert.calledWith(this.Analytics.sendEvent, "editor.instance.paste.addimage.cache");
    });

    test('Paste base64 image', function () {
        var $img = $('<img src="data:image/gif;base64,someencodeddata" />');
        this.$().is.withArgs('img').returns(true);
        this.$().find.withArgs('img:first-child:last-child').returns($img);
        this.$().match.withArgs(/^((data|blob):http)|(data:image)/).returns(true);
        this.$().on.withArgs('dropHandled.dropFileToElement').returns(true);
        this.pasteUtility.generateFileName.returns('generatedFilename');
        this.pasteUtility.dropFileToElement.returns(true);

        this.PastePlugin.configure(this.editorInstance, {
            plugins: {
                push: sinon.stub()
            }
        }).paste_postprocess(null, {
            content: ''
        });

        sinon.assert.calledOnce(this.pasteUtility.convertBlobToImage);
        var blob = this.pasteUtility.convertBlobToImage.getCall(0).args[0];
        equal(blob.type, "image/gif", 'proper content type should be set');
        sinon.assert.calledOnce(JIRA.Editor.Tags.Templates.image);
        sinon.assert.calledThrice(this.Analytics.sendEvent);
        sinon.assert.calledWith(this.Analytics.sendEvent, "editor.instance.paste");
        sinon.assert.calledWith(this.Analytics.sendEvent, "editor.instance.paste.image");
        sinon.assert.calledWith(this.Analytics.sendEvent, "editor.instance.paste.addimage.nocache.image");
    });

    test('Paste emoticon image', function () {
        this.$().is.withArgs('img').returns(true);
        var mock = _.extend({}, this.jQueryMethods, {0: {src: ""}});
        this.$().find.withArgs('img:first-child:last-child').returns(mock);
        this.$().hasClass.withArgs('emoticon').returns(true);
        this.Strings.startsWith.withArgs().returns(true);

        this.PastePlugin.configure(this.editorInstance, {
            plugins: {
                push: sinon.stub()
            }
        }).paste_postprocess(null, {
            content: ''
        });

        sinon.assert.notCalled(JIRA.Editor.Tags.Templates.image);
    });

    test('Paste not image', function () {
        this.$().is.withArgs('img').returns(false);

        this.PastePlugin.configure(this.editorInstance, {
            plugins: {
                push: sinon.stub()
            }
        }).paste_postprocess(null, {
            content: ''
        });

        sinon.assert.notCalled(JIRA.Editor.Tags.Templates.image);
    });

    module('PastePlugin - preprocessing', {
        setup: function () {
            this.context = AJS.test.mockableModuleContext();

            this.mockModules();

            this.PastePlugin = this.context.require('jira/editor/plugins/paste');
        },

        mockModules: function () {
            this.schema = {};

            this.EditorSchema = {
                sanitizeHtml: sinon.stub()
            };
            this.context.mock('jira/editor/schema', this.EditorSchema);

            this.pasteUtility = {
                convertBlobToImage: sinon.stub(),
                dropFileToElement: sinon.stub(),
                generateFileName: sinon.stub()
            };
            this.context.mock('dndattachment/ctrlv/utility', this.pasteUtility);

            this.editorInstance = {
                contextManager: {
                    getPasteSchemaForContext: function () {
                        return this.schema
                    }.bind(this)
                }
            };

            this.selectedNode = $("<div></div>");
        }
    });

    test('Should sanitizeHtml', function () {
        this.PastePlugin.configure(this.editorInstance, {
            plugins: {
                push: sinon.stub()
            }
        }).paste_preprocess(null, {
            content: ''
        });

        sinon.assert.calledWith(this.EditorSchema.sanitizeHtml, '', this.editorInstance.editor, this.schema);
    });
});
