define("jira/editor/plugins/mentions/mentions-util", [
    'jquery',
    'underscore'
], function ($,
             _
) {
    var ZERO_WIDTH_SPACE = 8203;
    var ZERO_WIDTH_SPACE_NO_BREAK = 65279;

    var stopPropagate = function (evt) {
        evt.preventDefault();
        evt.stopPropagation();
        evt.stopImmediatePropagation();
    };

    var mentionsUtil = {
        /**
         * Checks if passed node is mention node.
         * @param node
         * @returns {boolean}
         */
        isMentionNode: function (node) {
            if (!node) {
                return false;
            }
            return $(node).is('a.user-hover');
        },

        /**
         * Checks if carret is placed immediately after mention.
         * @param doc Document
         * @returns {boolean}
         */
        isCaretAfterMention: function (doc) {
            return this.getMentionBeforeCaret(doc) != null;
        },

        /**
         * Gets mention immediately before caret.
         * @param doc Document
         * @returns Node that is mention placed immediately before caret. null if caret is not placed after mention.
         */
        getMentionBeforeCaret: function (doc) {
            var prevNode = this.getPreviousNode(doc).node;
            if (this.isMentionNode(prevNode)) {
                return prevNode;
            }
            return null;
        },

        getPreviousNode: function (doc) {
            var getNodeOffset = function (node) {
                if (node.nodeType === Node.TEXT_NODE) {
                    return node.nodeValue.length;
                } else if (node.nodeType === Node.ELEMENT_NODE) {
                    return node.childNodes.length;
                } else {
                    return undefined;
                }
            };

            var selection = doc.getSelection();
            var node = selection.anchorNode;
            var offset = selection.anchorOffset;
            var defaultNode = {node: node, offset: selection.anchorOffset};

            if (selection.isCollapsed && node) {
                if (node.nodeType === Node.ELEMENT_NODE && node.childNodes) {
                    var idx = selection.anchorOffset - 1;
                    if (idx < 0) {
                        node = node.childNodes[0];
                        offset = 0;
                        var prevNode = this._getPreviousSiblingSkipEmptyTextNodes(node, offset);
                        if (prevNode) {
                            return {
                                node: prevNode,
                                offset: getNodeOffset(prevNode)
                            }
                        }
                    } else {
                        //sometimes chrome adds text node with 1 space after last contenteditable=false in line
                        var lastNodeRelaxedCheck = idx === node.childNodes.length - 1;
                        node = node.childNodes[idx];
                        offset = getNodeOffset(node);
                        var prevNode = this._getPreviousSiblingSkipEmptyTextNodes(node, offset, lastNodeRelaxedCheck);
                        if (prevNode) {
                            return {
                                node: prevNode,
                                offset: getNodeOffset(prevNode)
                            }
                        }
                    }
                } else if (node.nodeType === Node.TEXT_NODE) {
                    var prevNode = this._getPreviousSiblingSkipEmptyTextNodes(node, offset);
                    if (prevNode) {
                        return {
                            node: prevNode,
                            offset: prevNode === node ? offset : getNodeOffset(prevNode)
                        }
                    }
                }
            }
            return defaultNode;
        },

        _getPreviousSiblingSkipEmptyTextNodes: function (startNode, startNodeOffset, startNodeRelaxedCheck) {
            startNodeRelaxedCheck = startNodeRelaxedCheck || false;
            var prevNode = startNode;
            var outcome;
            while ( (outcome=this._isContentBeforeCursorEmpty(prevNode, startNodeOffset, startNodeRelaxedCheck)).outcome ) {
                prevNode = prevNode.previousSibling;
                startNodeOffset = undefined;
                startNodeRelaxedCheck = outcome.relaxed;
            }
            return prevNode;
        },

        _isContentBeforeCursorEmpty: function (node, offset, relaxedCheck) {
            var EMPTY = {
                outcome: true,
                relaxed: false
            };
            var NOT_EMPTY = {
                outcome: false,
                relaxed: false
            };
            var EMPTY_RELAXED = {
                outcome: true,
                relaxed: true
            };

            if (node && node.nodeType === Node.TEXT_NODE) {
                if (_.isUndefined(offset)) {
                    offset = node.nodeValue.length;
                }

                if (offset === 0) {
                    return EMPTY;
                }
                if (offset === 1 && this._isEmptyCharacter(node.nodeValue.charCodeAt(0)) && node.nodeValue.length === 1) {
                    return EMPTY;
                }
                if (relaxedCheck) {
                    if (offset === 1 && node.nodeValue === " ") {
                        return EMPTY_RELAXED;
                    }
                }
            }
            return NOT_EMPTY;
        },

        _isEmptyCharacter: function (ch) {
            return ch === ZERO_WIDTH_SPACE || ch === ZERO_WIDTH_SPACE_NO_BREAK;
        },

        /**
         * Selects mention node.
         * Current implementation assumes that caret is placed right after mention and creates fake LEFT keystroke to move caret left and select mention that is contenteditable=false.
         * @param mentionNode
         * @param evt
         */
        selectMention: function (mentionNode, evt, editor) {
            if (evt) {
                stopPropagate(evt);
            }
            editor.selection.select(mentionNode)
        }
    };

    return mentionsUtil;
});