define("jira/richeditor/editor-supported-operations", [
    "underscore"
], function (
    _
) {
    "use strict";

    var wikiOperations = ['bold', 'italic', 'underline', 'bullet-list', 'numbered-list',
        'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'paragraph', 'paragraph-quote', 'block-quote',
        'monospace', 'delete', 'superscript', 'subscript', 'cite', 'hr', 'color', 'icon',
        'attachment', 'code', 'panel', 'noformat', 'table', 'undo', 'redo'];

    var commonSupportedHtmlTags = ['br', 'span', 'a', 'p', 'div'];
    var requiredHtmlTags = mapWikiOperationToHtmlTags(wikiOperations, commonSupportedHtmlTags);

    return {
        getWikiOperations: function getWikiOperations() {
            return wikiOperations;
        },
        getRequiredHtmlTags: function getRequiredHtmlTags() {
            return requiredHtmlTags;
        },
        mapWikiOperationToHtmlTags: mapWikiOperationToHtmlTags
    };

    /**
     * Maps wiki operations to required html elements.
     * If element list is empty for an operation then no html tags will be added for that operation.
     * If the map does not contain particular key then it's assumed that such wiki
     *   operation has 1-1 html tag equivalent, eg. h1 or cite.
     * Adds "extraTags" to the list.
     *
     * @param {Array} operations
     * @param {Array=} extraTags list of html tags to be included in the result list
     * @returns List of html tags, without duplicates
     */
    function mapWikiOperationToHtmlTags(operations, extraTags) {
        var map = {
            'bold': ['strong', 'b'],
            'italic': ['em', 'i'],
            'underline': ['ins', 'u'],
            'bullet-list': ['li', 'ul'],
            'numbered-list': ['li', 'ol'],
            'paragraph': [],
            'paragraph-quote': ['blockquote'],
            'block-quote': ['blockquote'],
            'monospace': ['samp', 'tt', 'code'],
            'delete': ['del'],
            'superscript': ['sup'],
            'subscript': ['sub'],
            'color': ['font'],
            'icon': ['img'],
            'attachment': ['a', 'img', 'sup'],
            'code': ['pre'],
            'panel': ['b'],
            'noformat': ['pre'],
            'table': ['table', 'thead', 'tbody', 'tr', 'th', 'td', 'tfoot', 'caption', 'colgroup', 'col'],
            'undo': [],
            'redo': []
        };

        var tagList = _.map(operations, function (operation) {
            return map[operation] || [operation];
        });
        tagList.push(extraTags || []);
        var tags = _.flatten(tagList);
        return _.uniq(tags);
    }
});