AJS.test.require(["com.atlassian.jira.plugins.jira-editor-plugin:schema-builder"], function () {
    "use strict";

    var _ = require("underscore");

    /**
     * Call it with the module context.
     */
    function setup() {
        this.context = AJS.test.mockableModuleContext();
        this.sandbox = sinon.sandbox.create();

        this.logger = {
            error: sinon.spy()
        };

        this.context.mock("jira/util/logger", this.logger);
        this.SchemaBuilder = new (this.context.require('jira/editor/schema-builder'));
        this.elements = this.SchemaBuilder.customElementChildren;
        this.attrs = this.SchemaBuilder.customElementAttrs;
    }

    module("SchemaBuilder - withCustomElement with correct params", {
        setup: function () {
            setup.call(this);
        },
        teardown: function () {
            this.sandbox.restore();
        }
    });


    test('Should be able to handle an empty spec', function () {
        ok(Object.keys(this.elements).length === 0, 'Spec should be empty since no element was added');
    });

    test('Should add one element with default children and no attributes', function () {
        this.SchemaBuilder.withCustomElement('x-fake-element');
        deepEqual(this.elements, {'x-fake-element': ['p', '#comment']});
        deepEqual(this.attrs, {'x-fake-element': []});
    });

    test('Should add one element with no children and attributes', function () {
        this.SchemaBuilder.withCustomElement('x-fake-element', false);
        deepEqual(this.elements, {'x-fake-element': []});
        deepEqual(this.attrs, {'x-fake-element': []});
    });

    test('Should add one element with specified children and no attributes', function () {
        this.SchemaBuilder.withCustomElement('x-fake-element', ['p', 'div', 'table']);
        deepEqual(this.elements, {'x-fake-element': ['p', 'div', 'table', '#comment']});
        deepEqual(this.attrs, {'x-fake-element': []});
    });

    test('Should add one element with specified attributes and no children', function () {
        this.SchemaBuilder.withCustomElement('x-fake-element', [], ['a1', 'a2']);
        deepEqual(this.elements, {'x-fake-element': ['#comment']});
        deepEqual(this.attrs, {'x-fake-element': ['a1', 'a2']});
    });

    test('Should add two elements', function () {
        this.SchemaBuilder.withCustomElement('x-fake-element', ['p', 'div', '#text']);
        this.SchemaBuilder.withCustomElement('x-fake-list', ['ol', 'ul'], ['a1']);
        deepEqual(this.elements, {
            'x-fake-element': ['p', 'div', '#text', '#comment'],
            'x-fake-list': ['ol', 'ul', '#comment']
        });
        deepEqual(this.attrs, {'x-fake-element': [], 'x-fake-list': ['a1']});
    });

    test('Should remove duplicated "#comment" child', function () {
        this.SchemaBuilder.withCustomElement('x-fake-element', ['#comment']);
        deepEqual(this.elements, {'x-fake-element': ['#comment']});
        deepEqual(this.attrs, {'x-fake-element': []});
    });

    test('Should remove duplicated children', function () {
        this.SchemaBuilder.withCustomElement('x-fake-element', ['p', '#text', 'p', 'p', '#text']);
        deepEqual(this.elements, {'x-fake-element': ['p', '#text', '#comment']});
        deepEqual(this.attrs, {'x-fake-element': []});
    });

    test('Should remove duplicated attributes', function () {
        this.SchemaBuilder.withCustomElement('x-fake-element', [], ['a1', 'a2', 'a1', 'a1', 'a2']);
        deepEqual(this.elements, {'x-fake-element': ['#comment']});
        deepEqual(this.attrs, {'x-fake-element': ['a1', 'a2']});
    });

    module("SchemaBuilder - withCustomElement with incorrect params", {
        setup: function () {
            setup.call(this);
        },
        teardown: function () {
            this.sandbox.restore();
        }
    });

    test('Should not add element when its name does not contain dash "-"', function () {
        this.SchemaBuilder.withCustomElement('xfakeelement');
        sinon.assert.calledWith(this.logger.error, sinon.match('incorrect `name`'));
    });

    test('Should not add element when its name is empty', function () {
        this.SchemaBuilder.withCustomElement('');
        sinon.assert.calledWith(this.logger.error, sinon.match('incorrect `name`'));
    });

    test('Should not add element when its name is not a string #1', function () {
        this.SchemaBuilder.withCustomElement([]);
        sinon.assert.calledWith(this.logger.error, sinon.match('incorrect `name`'));
    });

    test('Should not add element when its name is not a string #2', function () {
        this.SchemaBuilder.withCustomElement(true);
        sinon.assert.calledWith(this.logger.error, sinon.match('incorrect `name`'));
    });

    test('Should not add element when its name is not a string #3', function () {
        this.SchemaBuilder.withCustomElement({});
        sinon.assert.calledWith(this.logger.error, sinon.match('incorrect `name`'));
    });

    test('Should not add element when `children` arg is equal null', function () {
        this.SchemaBuilder.withCustomElement('x-fake', null);
        sinon.assert.calledWith(this.logger.error, sinon.match('incorrect `children`'));
    });

    test('Should not add element when `children` arg is equal true', function () {
        this.SchemaBuilder.withCustomElement('x-fake', true);
        sinon.assert.calledWith(this.logger.error, sinon.match('incorrect `children`'));
    });

    test('Should not add element when `children` arg is an object', function () {
        this.SchemaBuilder.withCustomElement('x-fake', {});
        sinon.assert.calledWith(this.logger.error, sinon.match('incorrect `children`'));
    });

    test('Should not add element when `children` arg is not array of strings #1', function () {
        this.SchemaBuilder.withCustomElement('x-fake', [1]);
        sinon.assert.calledWith(this.logger.error, sinon.match('incorrect `children`'));
    });

    test('Should not add element when `children` arg is not array of strings #2', function () {
        this.SchemaBuilder.withCustomElement('x-fake', ['p', null]);
        sinon.assert.calledWith(this.logger.error, sinon.match('incorrect `children`'));
    });

    test('Should not add element when `children` arg is not array of strings #3', function () {
        this.SchemaBuilder.withCustomElement('x-fake', ['p', 'div', ['p', 'b']]);
        sinon.assert.calledWith(this.logger.error, sinon.match('incorrect `children`'));
    });

    test('Should not add element when `attributes` arg is equal null', function () {
        this.SchemaBuilder.withCustomElement('x-fake', [], null);
        sinon.assert.calledWith(this.logger.error, sinon.match('incorrect `attributes`'));
    });

    test('Should not add element when `attributes` arg is equal true', function () {
        this.SchemaBuilder.withCustomElement('x-fake', [], true);
        sinon.assert.calledWith(this.logger.error, sinon.match('incorrect `attributes`'));
    });

    test('Should not add element when `attributes` arg is an object', function () {
        this.SchemaBuilder.withCustomElement('x-fake', [], {});
        sinon.assert.calledWith(this.logger.error, sinon.match('incorrect `attributes`'));
    });

    test('Should not add element when `attributes` arg is not array of strings #1', function () {
        this.SchemaBuilder.withCustomElement('x-fake', [], [1]);
        sinon.assert.calledWith(this.logger.error, sinon.match('incorrect `attributes`'));
    });

    test('Should not add element when `attributes` arg is not array of strings #2', function () {
        this.SchemaBuilder.withCustomElement('x-fake', [], ['p', null]);
        sinon.assert.calledWith(this.logger.error, sinon.match('incorrect `attributes`'));
    });

    test('Should not add element when `attributes` arg is not array of strings #3', function () {
        this.SchemaBuilder.withCustomElement('x-fake', [], ['p', 'div', ['p', 'b']]);
        sinon.assert.calledWith(this.logger.error, sinon.match('incorrect `attributes`'));
    });

    module("SchemaBuilder - build", {
        setup: function () {
            setup.call(this);
        },
        teardown: function () {
            this.sandbox.restore();
        }
    });

    test('Should produce proper extended_valid_elements spec for custom element with no attributes', function() {
        var spec = this.SchemaBuilder
            .withCustomElement('x-fake')
            .build();

        strictEqual(spec.getExtendedValidElements(), 'x-fake');
    });

    test('Should produce proper extended_valid_elements spec for custom element with one attribute', function() {
        var spec = this.SchemaBuilder
            .withCustomElement('x-fake', [], ['a1'])
            .build();

        strictEqual(spec.getExtendedValidElements(), 'x-fake[a1]');
    });

    test('Should produce proper extended_valid_elements spec for custom element with three attributes', function() {
        var spec = this.SchemaBuilder
            .withCustomElement('x-fake', [], ['a1', 'a2', 'a3'])
            .build();

        strictEqual(spec.getExtendedValidElements(), 'x-fake[a1|a2|a3]');
    });

    test('Should produce proper extended_valid_elements spec for custom elements with no attributes', function() {
        var spec = this.SchemaBuilder
            .withCustomElement('x-fake')
            .withCustomElement('x-list')
            .build();

        strictEqual(spec.getExtendedValidElements(), 'x-fake,x-list');
    });

    test('Should produce proper extended_valid_elements spec for custom elements with attributes', function() {
        var spec = this.SchemaBuilder
            .withCustomElement('x-fake', [], ['a1'])
            .withCustomElement('x-list', [], ['a2', 'a3'])
            .build();

        strictEqual(spec.getExtendedValidElements(), 'x-fake[a1],x-list[a2|a3]');
    });


    module("SchemaBuilder - reusing builder", {
        setup: function () {
            setup.call(this);
        },
        teardown: function () {
            this.sandbox.restore();
        }
    });

    test('Two specs built from the same builder should be independent of each other', function() {
        this.SchemaBuilder
            .setValidChildren('div', 'span');
        var builder1 = this.SchemaBuilder.build();

        this.SchemaBuilder
            .withBaseSchema('123')
            .withCustomElement('x-fake')
            .setValidChildren('p', 'span')
            .withValidElements(['div']);
        var builder2 = this.SchemaBuilder.build();

        ok(builder1.getType() !== builder2.getType(), 'output of `getType` should not be the same');
        ok(builder1.getValidChildren() !== builder2.getValidChildren(), 'output of `getValidChildren` should not be the same');
        ok(builder1.getCustomElements() !== builder2.getCustomElements(), 'output of `getCustomElements` should not be the same');
        ok(builder1.getExtendedValidElements() !== builder2.getExtendedValidElements(), 'output of `getExtendedValidElements` should not be the same');
        ok(builder1.getValidElements() !== builder2.getValidElements(), 'output of `getValidElements` should not be the same');
    });
});