define("jira/editor/create", [
    "jira/editor/instance",
    "jira/editor/resources",
    "jira/editor/schema",
    "jira/editor/customizer",
    "jira/editor/height-helper",
    "jira/util/navigator",
    "jira/editor/tinymce",
    "jquery",
    "underscore"
], function (EditorInstance,
             EditorResources,
             EditorSchema,
             EditorCustomizer,
             HeightHelper,
             Navigator,
             tinymce,
             $,
             _) {
    var Builder = function (element) {
        this.element = element;
        this.settings = {
            toolbar: false,
            menubar: false,
            elementpath: false
        };
        this.plugins = [];
    };

    Builder.prototype.build = function () {
        var deferred = new $.Deferred();
        var that = this;

        var instance = new EditorInstance(that.element);

        var heightHelper = new HeightHelper(this.element);
        // tinymce adds additional border of size 2, we have to remove it from textarea height to prevent jumping
        var heightOffset = -2;

        var mceSettings = _.extend(this.settings, {
            plugins: ["autoresize", "lists", "table"],

            // this is for convenience, in editor-i18n.js we will override this language translation
            // this is needed, because we don't want to download this resource asynchronously at this point
            language: "en",

            content_style: Navigator.isIE() ? "body#tinymce {min-height: 10px}" : "",

            autoresize_on_init: true,
            autoresize_max_height: heightHelper.getMaxHeight() + heightOffset,
            autoresize_min_height: heightHelper.getMinHeight() + heightOffset,
            autoresize_bottom_margin: 0,

            allow_conditional_comments: false,

            // object resizing not supported by wiki markup for emoticons
            object_resizing: "img:not(.emoticon)",

            end_container_on_empty_block: true,

            browser_spellcheck: true,

            table_toolbar: "tabledelete | tableinsertrowbefore tableinsertrowafter tabledeleterow | tableinsertcolbefore tableinsertcolafter tabledeletecol",

            hidden_input: false,
            target: this.element,

            autoresize_overflow_padding: 8,

            relative_urls: false,
            //MNSTR-1559 tinymce (tested 4.7.13) does not handle properly ipv6 addresses in pasted links eg.
            //http://[2600:1f18:674c:af02:acbf:31b:c7fe:1883]:8080/browse/LP-1 becomes http://[2600:1f18:674c:af02:acbf:31b:c7fe:1883]:8080
            //due to bug in tinymce URI class (https://github.com/tinymce/tinymce/issues/4409)
            convert_urls: false,

            branding: false,

            formats: {
                underline: [
                    {inline: 'ins', remove: 'all'},
                    {inline: 'u', remove: 'all'},
                    {inline: 'span', styles: {textDecoration: 'underline'}, exact: true}
                ],
                strikethrough: [
                    {inline: 'del', remove: 'all'},
                    {inline: 'strike', remove: 'all'},
                    {inline: 'span', styles: {textDecoration: 'line-through'}, exact: true}
                ]
            },

            setup: function (editor) {
                deferred.notify(instance);
                instance.init(editor);
                editor.on("init", function () {
                    editor.getContainer().classList.add('jira-editor-container');
                    instance.schemaSpec.fixChildren(editor.schema);

                    deferred.resolve(instance);
                });

                // required to hide i.e. AUI Dropdown
                editor.on("focus", function () {
                    $(that.element).trigger("click").trigger("focusin");
                });
            }
        });

        $.when(
            EditorResources.loadCssResources(),
            EditorResources.loadPlugins(this.plugins)
        ).then(function (cssResources) {
            this.plugins.forEach(function (plugin) {
                require(plugin.name).configure(instance, mceSettings); // eslint-disable-line module-checks/no-dynamic-require
            }, this);

            var schemaSpecBuilder = EditorSchema.getSchemaSpecBuilder();
            EditorCustomizer.getCustomizeSettingCallbacks().forEach(function (callback) {
                callback(mceSettings, tinymce, schemaSpecBuilder);
            });

            setSchemas(schemaSpecBuilder, mceSettings, instance);

            mceSettings.content_css = cssResources;

            // should be fixed in tinymce :(
            if (document.readyState === "interactive") {
                tinymce.dom.EventUtils.Event.domLoaded = true;
            }

            // there might be situation when we initialize editor (start downloading resources),
            // then lost focus on editor so it disappears, resources are downloaded and editor tries to initialize and blows
            // it should be fixed in tinymce, but bug report exists from Jul 20, 2015
            // https://github.com/tinymce/tinymce/issues/2319
            if (document.body.contains(this.element)) {
                tinymce.init(mceSettings);
            }
            else {
                deferred.reject();
            }
        }.bind(this));

        return deferred.promise();
    };

    Builder.prototype.withElement = function (element) {
        this.element = element;

        return this;
    };

    Builder.prototype.withPlugin = function (pluginKey, pluginResource) {
        this.plugins.push({
            name: pluginKey,
            resource: pluginResource
        });

        return this;
    };

    Builder.prototype.withMeta = function (meta) {
        this.meta = meta;

        return this;
    };

    return {
        builder: function (element) {
            return new Builder(element);
        }
    };


    function setSchemas(schemaSpecBuilder, mceSettings, instance) {
        var schemaSpec = schemaSpecBuilder.build();
        mceSettings.schema = schemaSpec.getType();
        mceSettings.custom_elements = schemaSpec.getCustomElements();
        mceSettings.valid_children = schemaSpec.getValidChildren();
        mceSettings.valid_elements = schemaSpec.getValidElements();
        mceSettings.extended_valid_elements = schemaSpec.getExtendedValidElements();

        instance.schemaSpec = schemaSpec;
        instance.pasteSchemaSpec = EditorSchema.getPasteSchemaSpecBuilder(schemaSpecBuilder).build();
        instance.pasteInsidePreSchemaSpec = EditorSchema.getPasteInsidePreSchemaSpecBuilder().build();
    }
});
