AJS.test.require([
    'com.atlassian.jira.plugins.jira-editor-plugin:constants',
    'com.atlassian.jira.plugins.jira-editor-plugin:api',
    'com.atlassian.jira.plugins.jira-editor-plugin:registry'
], function () {
    var Entry = require('jira/editor/registry/entry');
    var Modes = require('jira/editor/constants').Modes;
    var EditorInstance = require('jira/editor/instance');
    var editorRegistry = require('jira/editor/registry');

    module('EditorRegistry', {
        getExpectedEntry: function (mode) {
            return {
                id: this.id,
                toolbar: this.toolbar,
                textArea: this.textArea,
                currentMode: mode
            }
        },

        setup: function () {
            this.sandbox = sinon.sandbox.create();
            this.textArea = this.sandbox.mock();
            this.toolbar = this.sandbox.mock();
            this.id = Date.now().toString();
            this.sandbox.stub(Entry.prototype, '_findToolbar').returns(this.toolbar);
            this.sandbox.stub(editorRegistry, '_findTextArea').returns(this.textArea);
            this.sandbox.stub(editorRegistry, '_getTextAreaId').returns(this.id);
            this.sandbox.spy(editorRegistry, '_triggerEvents');
        },

        teardown: function () {
            this.sandbox.restore();
            editorRegistry._instances = {};
        }
    });

    test("Should trigger `register` events on Text mode activation", function () {
        editorRegistry._registerTextMode(this.textArea);

        var expectedEntry = this.getExpectedEntry(Modes.TEXT);

        var triggerEvents = editorRegistry._triggerEvents;
        ok(triggerEvents.calledOnce, "_triggerEvents called once");
        ok(triggerEvents.firstCall.calledWithMatch(sinon.match('register'), matchEntry(expectedEntry)),
            "_triggerEvents called with correct arguments");
    });

    test("Should trigger `register` events on Visual mode activation", function (assert) {
        var editorInstance = new EditorInstance();
        editorRegistry._registerVisualMode(editorInstance);

        var expectedEntry = this.getExpectedEntry(Modes.VISUAL);

        var triggerEvents = editorRegistry._triggerEvents;
        ok(triggerEvents.calledOnce, "_triggerEvents called once");
        ok(triggerEvents.firstCall.calledWithMatch(sinon.match('register'), matchEntry(expectedEntry)),
            "_triggerEvents called with correct arguments");
    });

    test("Should register correct entry in Text mode", function () {
        editorRegistry._registerTextMode(this.textArea);

        var expectedEntry = this.getExpectedEntry(Modes.TEXT);

        var actualEntry = editorRegistry.getEntry(this.id);
        ok(sinon.match(actualEntry, matchEntry(expectedEntry)), "editorRegistry contains correct entry");
    });

    asyncTest("Should register correct entry in Visual mode", function () {
        var editorInstane = new EditorInstance();
        editorRegistry._registerVisualMode(editorInstane);

        var expectedEntry = this.getExpectedEntry(Modes.VISUAL);

        var actualEntry = editorRegistry.getEntry(this.id);
        ok(sinon.match(actualEntry, matchEntry(expectedEntry)), "editorRegistry contains correct entry");

        actualEntry.rteInstance.done(function (actualRTEInstance) {
            start();
            strictEqual(actualRTEInstance, editorInstane, "Entry has correctly resolved rteInstance");
        });
    });

    test("Should trigger `unregister` events upon unregistration in Text mode", function () {
        editorRegistry._registerTextMode(this.textArea);
        editorRegistry._unregister(this.textArea);

        var expectedEntry = this.getExpectedEntry(Modes.TEXT);

        var triggerEvents = editorRegistry._triggerEvents;
        ok(triggerEvents.calledTwice, "_triggerEvents called twice");
        ok(triggerEvents.secondCall.calledWithMatch(sinon.match('unregister'), matchEntry(expectedEntry)),
            "_triggerEvents called with correct arguments");
    });

    test("Should trigger `unregister` events upon unregistration in Visual mode", function () {
        editorRegistry._registerVisualMode(new EditorInstance());
        editorRegistry._unregister(this.textArea);

        var expectedEntry = this.getExpectedEntry(Modes.VISUAL);

        var triggerEvents = editorRegistry._triggerEvents;
        ok(triggerEvents.calledTwice, "_triggerEvents called twice");
        ok(triggerEvents.secondCall.calledWithMatch(sinon.match('unregister'), matchEntry(expectedEntry)),
            "_triggerEvents called with correct arguments");
    });

    test("Should trigger `unregister` events upon unregistration after switching modes", function () {
        editorRegistry._registerTextMode(this.textArea);
        editorRegistry._registerVisualMode(new EditorInstance());
        editorRegistry._switchMode(this.textArea, Modes.TEXT);
        editorRegistry._unregister(this.textArea);

        var expectedEntry = this.getExpectedEntry(Modes.TEXT);

        var triggerEvents = editorRegistry._triggerEvents;
        equal(triggerEvents.callCount, 4, "_triggerEvents called 4 times");
        ok(triggerEvents.lastCall.calledWithMatch(sinon.match('unregister'), matchEntry(expectedEntry)),
            "_triggerEvents called with correct arguments");
    });

    test("Should remove entry correctly upon unregistration in Text mode", function () {
        editorRegistry._registerTextMode(this.textArea);
        editorRegistry._unregister(this.textArea);

        var actualEntry = editorRegistry.getEntry(this.id);
        strictEqual(actualEntry, undefined, "editorRegistry has entry removed after `unregister` event");
    });

    test("Should remove entry correctly upon unregistration in Visual mode", function () {
        editorRegistry._registerVisualMode(new EditorInstance());
        editorRegistry._unregister(this.textArea);

        var actualEntry = editorRegistry.getEntry(this.id);
        strictEqual(actualEntry, undefined, "editorRegistry has entry removed after `unregister` event");
    });

    test("Should remove entry correctly upon unregistration after switching modes", function () {
        editorRegistry._registerTextMode(this.textArea);
        editorRegistry._registerVisualMode(new EditorInstance());
        editorRegistry._switchMode(this.textArea, Modes.TEXT);
        editorRegistry._unregister(this.textArea);

        var actualEntry = editorRegistry.getEntry(this.id);
        strictEqual(actualEntry, undefined, "editorRegistry has entry removed after `unregister` event");
    });

    test("Should trigger `switch` events on the first Text->Visual switch (via `_registerVisualMode` call)", function () {
        editorRegistry._registerTextMode(this.textArea);
        editorRegistry._registerVisualMode(new EditorInstance());

        var expectedEntry = this.getExpectedEntry(Modes.VISUAL);

        var triggerEvents = editorRegistry._triggerEvents;
        ok(triggerEvents.calledTwice, "_triggerEvents called twice");
        ok(triggerEvents.secondCall.calledWithMatch(sinon.match('switch'), matchEntry(expectedEntry)),
            "_triggerEvents called with correct arguments");
    });

    test("Should trigger `switch` events when repetitively switching modes", function () {
        editorRegistry._registerTextMode(this.textArea);

        var switches = 10;
        for (var i = 0; i < switches; i++) {
            var mode = (i % 2 === 0) ? Modes.VISUAL : Modes.TEXT;
            editorRegistry._switchMode(this.textArea, mode);
        }

        var expectedEntry = this.getExpectedEntry(Modes.TEXT);

        var triggerEvents = editorRegistry._triggerEvents;
        ok(triggerEvents.callCount, switches + 1, "_triggerEvents called " + (switches + 1) + " times");
        ok(triggerEvents.firstCall.calledWithMatch(sinon.match('register'), matchEntry(expectedEntry)),
            "_triggerEvents called with correct arguments");
        for (i = 0; i < switches; i++) {
            ok(triggerEvents.calledWith('switch'), "_triggerEvents called for `switch` #" + (i + 1) );
        }
    });

    test("Should NOT trigger `switch` events when calling `_switch` passing the current mode", function () {
        editorRegistry._registerTextMode(this.textArea);
        editorRegistry._switchMode(this.textArea, Modes.TEXT);

        var expectedEntry = this.getExpectedEntry(Modes.TEXT);

        var triggerEvents = editorRegistry._triggerEvents;
        ok(triggerEvents.calledOnce, "_triggerEvents called once");
        ok(triggerEvents.firstCall.calledWithMatch(sinon.match('register'), matchEntry(expectedEntry)),
            "_triggerEvents called with correct arguments");
        ok(triggerEvents.neverCalledWith('switch'), "_triggerEvents not called for `switch` events");
    });

    function matchEntry(entry) {
        return function (arg) {
            for (var key in entry) {
                if (entry.hasOwnProperty(key) && entry[key] !== arg[key]) {
                    return false;
                }
            }
            return true;
        }
    }
});