AJS.test.require([
    "com.atlassian.jira.plugins.jira-editor-plugin:undo-redo-ui",
    "com.atlassian.jira.plugins.jira-editor-plugin:undo-redo",
    "jquery"
], function () {
    "use strict";

    var $ = require("jquery");
    var EditorConstants = require("jira/editor/constants");
    var UndoRedoButtons = require("jira/editor/ui/undo-redo-buttons");
    var NoOpAdapter = require("jira/editor/undomanager-adapter/noop-adapter");

    NoOpAdapter.prototype.triggerUndo = function() {
        this.trigger("editor:historychanged", EditorConstants.EventTypes.UNDO, EditorConstants.EventSources.BUTTON, "");
    };
    NoOpAdapter.prototype.triggerRedo = function() {
        this.trigger("editor:historychanged", EditorConstants.EventTypes.REDO, EditorConstants.EventSources.BUTTON, "");
    };

    UndoRedoButtons.prototype._clickUndo = function() {
        this.ui.undo.click();
    };
    UndoRedoButtons.prototype._clickRedo = function() {
        this.ui.redo.click();
    };
    UndoRedoButtons.prototype._isVisible = function() {
        return this.$el.is(":visible");
    };
    UndoRedoButtons.prototype._isUndoEnabled = function() {
        return !this.ui.undo.is(":disabled");
    };
    UndoRedoButtons.prototype._isRedoEnabled = function() {
        return !this.ui.redo.is(":disabled");
    };
    UndoRedoButtons.prototype._hasDivider = function() {
        return this.$el.hasClass("wiki-edit-separate-section");
    };

    /**
     * Should be called with module context.
     * @param {jQuery} $markup valid markup containing textarea and desired container for buttons
     */
    function setup($markup) {
        this.sandbox = sinon.sandbox.create();

        var textarea = $markup.find("textarea");
        $("#qunit-fixture").append($markup);
        this.textarea = textarea;
        this.sandbox.spy(this.textarea, "focus");

        this.undoManager = new NoOpAdapter();

        this.view = new UndoRedoButtons({
            textarea: textarea,
            undoManager: this.undoManager
        });

        this.sandbox.spy(this.undoManager, "undo");
        this.sandbox.spy(this.undoManager, "redo");
        this.sandbox.spy(this.view, "updateUI");

        this.view.render();
    }

    module("Undo Redo Buttons - general", {
        setup: function() {
            this.$markup = $('<div class="jira-wikifield">\n    <div>\n        <textarea></textarea>\n    </div>\n</div>\n<div class="save-options"><div class="wiki-button-bar-content">\n    <button></button>\n</div></div>');
            setup.call(this, this.$markup);
        },
        teardown: function() {
            this.$markup.remove();
            this.sandbox.restore();
        }
    });

    test("Should attach buttons (create issue context)", function() {
        ok(this.view._isVisible());
    });

    test("Should not show divider (create issue context)", function() {
        ok(!this.view._hasDivider());
    });

    test("Should update UI on undo", function() {
        this.undoManager.triggerUndo();
        ok(this.view.updateUI.called, "view.updateUI should be called");
    });

    test("Should update UI on redo", function() {
        this.undoManager.triggerRedo();
        ok(this.view.updateUI.called, "view.updateUI should be called");
    });

    test("Should disable undo button when there is no undo levels", function() {
        this.sandbox.stub(this.undoManager, 'hasUndo').returns(true);
        this.view.updateUI();

        ok(this.view._isUndoEnabled(), "undo button should be enabled");

        this.undoManager.hasUndo.returns(false);
        this.view.updateUI();

        ok(!this.view._isUndoEnabled(), "undo button should be disabled");
    });

    test("Should disable redo button when there is no redo levels", function() {
        this.sandbox.stub(this.undoManager, 'hasRedo').returns(true);
        this.view.updateUI();

        ok(this.view._isRedoEnabled(), "redo button should be enabled");

        this.undoManager.hasRedo.returns(false);
        this.view.updateUI();

        ok(!this.view._isRedoEnabled(), "redo button should be disabled");
    });

    test("Should not undo when button is disabled", function() {
        this.view._clickUndo();
        ok(!this.undoManager.undo.called, "undoManager.undo should not be called");
    });

    test("Should undo when button is enabled", function() {
        this.sandbox.stub(this.undoManager, 'hasUndo').returns(true);
        this.view.updateUI();

        this.view._clickUndo();
        ok(this.undoManager.undo.called, "undoManager.undo should be called");
        ok(this.textarea.focus.called, "textarea.focus should be called");

        this.view._clickUndo();
        ok(this.undoManager.undo.calledTwice, "undoManager.undo should be called twice");
        ok(this.textarea.focus.calledTwice, "textarea.focus should be called twice");
    });

    test("Should not redo when button is disabled", function() {
        this.view._clickRedo();
        ok(!this.undoManager.redo.called, "undoManager.redo should not be called");
    });

    test("Should redo when button is enabled", function() {
        this.sandbox.stub(this.undoManager, 'hasRedo').returns(true);
        this.view.updateUI();

        this.view._clickRedo();
        ok(this.undoManager.redo.called, "undoManager.redo should be called");
        ok(this.textarea.focus.called, "textarea.focus should be called");

        this.view._clickRedo();
        ok(this.undoManager.redo.calledTwice, "undoManager.redo should be called twice");
        ok(this.textarea.focus.calledTwice, "textarea.focus should be called twice");
    });


    module("Undo Redo Buttons - add and edit comment with lock", {
        setup: function() {
            this.$markup = $('<div class="jira-wikifield">\n    <div>\n        <textarea></textarea>\n    </div>\n</div>\n<div class="save-options"><div class="wiki-button-bar-content">\n    <a>lock</a>\n    <input>\n</div></div>');
            setup.call(this, this.$markup);
        },
        teardown: function() {
            this.$markup.remove();
            this.sandbox.restore();
        }
    });

    test("Should attach buttons", function() {
        ok(this.view._isVisible());
    });

    test("Should show divider", function() {
        ok(this.view._hasDivider());
    });

    module("Undo Redo Buttons - add and edit comment without lock", {
        setup: function() {
            this.$markup = $('<div class="jira-wikifield">\n    <div>\n        <textarea></textarea>\n    </div>\n</div>\n<div class="save-options"><div class="wiki-button-bar-content">\n   <input>\n</div></div>');
            setup.call(this, this.$markup);
        },
        teardown: function() {
            this.$markup.remove();
            this.sandbox.restore();
        }
    });

    test("Should attach buttons", function() {
        ok(this.view._isVisible());
    });

    test("Should not show divider", function() {
        ok(!this.view._hasDivider());
    });

    module("Undo Redo Buttons - inline edit", {
        setup: function() {
            this.$markup = $('<div class="inline-edit-fields">\n    <div>\n        <textarea></textarea>\n    </div>\n</div>\n<div></div>\n<div class="save-options"><div class="wiki-button-bar-content">\n    <button></button>\n</div></div>');
            setup.call(this, this.$markup);
        },
        teardown: function() {
            this.$markup.remove();
            this.sandbox.restore();
        }
    });

    test("Should attach buttons", function() {
        ok(this.view._isVisible());
    });

    test("Should not show divider", function() {
        ok(!this.view._hasDivider());
    });

});