define("jira/editor/ui/rich-editor", [
    'atlassian/libs/skate-0.12.6',
    'jira/util/logger',
    'jira/util/navigator',
    'jira/editor/controller',
    'jira/editor/analytics',
    'jira/editor/round-trip',
    'jira/editor/fsm',
    'jira/editor/constants',
    'jquery',
    'underscore'
], function(
    skate,
    logger,
    Navigator,
    controller,
    Analytics,
    RoundTrip,
    FSM,
    Constants,
    $,
    _
) {
    // this resource may come from either plugin or extension, so we allow only one init to take control
    if (window.JIRA_EDITOR_CONNECT_PLUGIN_INSTALLED ||
        // MSIE below 11 is not supported
        Navigator.isIE() && !Navigator.isEdge() && Navigator.majorVersion() < 11
    ) {
        return;
    }
    window.JIRA_EDITOR_CONNECT_PLUGIN_INSTALLED = true;

    $.valHooks.textarea = {
        get: function (el) {
        },

        set: function (el, value) {
            $(el).trigger(Constants.EVENT_CHANGE, value);
        }
    };

    controller.overrideWikiPreview();

    var PROPERTY = "richEditorInstance";
    var PROPERTY_LOADING = "richEditorInstance_loading";

    var skateHandler = {
        prototype: {
            showMask: function() {
                $(this).addClass(Constants.CLASS_LOADING);
                this.getToolbar().addClass(Constants.CLASS_LOADING);
                this.getContainer().addClass(Constants.CLASS_LOADING);
                this.getInstance() && this.getInstance().disable();
            },

            hideMask: function() {
                $(this).removeClass(Constants.CLASS_LOADING);
                this.getToolbar().removeClass(Constants.CLASS_LOADING);
                this.getContainer().removeClass(Constants.CLASS_LOADING);
                this.getInstance() && this.getInstance().enable();
            },

            getToolbar: function() {
                return this.getToolbarContainer().find(".wiki-edit-toolbar");
            },

            getContainer: function () {
                return $(this.parentNode);
            },

            getToolbarContainer: function() {
                return $(this).parents('.wiki-edit, .js-comment-form, .servicedesk-attachments-dialog, .sd-comment-field-form-container');
            },

            coverTextarea: function() {
                this.$textarea
                    .attr("tabindex", -1)
                    .addClass(Constants.CLASS_COVER)
                    .on('focus.rteCover', function (e) {
                        e.preventDefault();
                    });
            },

            uncoverTextarea: function() {
                this.$textarea
                    .removeAttr("tabindex")
                    .removeClass(Constants.CLASS_COVER)
                    .off('.rteCover');

                //triggers expanding height from expandOnInput.js
                if( this.$textarea.expandOnInput ) {
                    this.$textarea.css({'height' : ''});
                    this.$textarea.expandOnInput();
                }
            },

            getRenderParams: function() {
                return $(this).data('renderParams');
            },

            replaceSelection: function(markup) {
                controller.renderMarkup(markup, this.getRenderParams()).done(function(rendered) {
                    this.getInstance().replaceSelection(rendered);
                }.bind(this));
            },

            syncSelection: function(textarea) {
                var start = textarea.selectionStart;
                var end = textarea.selectionEnd;
                var length = textarea.value.length;
                var instance = this.getInstance();
                _.defer(function () {
                    if (end === length) {
                        if (start === 0) {
                            instance.selectAll();
                        } else if (start === end) {
                            // move cursor to end
                            instance.selectAll();
                            instance.editor.selection.collapse(false);
                        }
                    }

                    instance.focus();
                });
            },

            setContent: function(markup) {
                return controller.renderMarkup(markup, this.getRenderParams()).done(function(rendered) {
                    this._setRawContent(rendered);
                }.bind(this));
            },

            /**
             * @param rendered Rendered content
             * @param silent Do not trigger event listeners
             * @private
             */
            _setRawContent: function(rendered, silent) {
                this.getInstance().setContent(rendered, null, silent);
            },

            getContent: function () {
                return this.$textarea.val();
            },

            focus: function() {
                this.getInstance().focus();
            },

            setMode: function(showSource) {
                var result = new $.Deferred();

                if (showSource) {
                    Analytics.sendEvent("editor.instance.switch.source");
                    Analytics.sendEvent("bundled.editor.instance.switch.source");
                } else {
                    Analytics.sendEvent("editor.instance.switch.wysiwyg");
                    Analytics.sendEvent("bundled.editor.instance.switch.wysiwyg");
                }

                if (showSource) {
                    this.uncoverTextarea();
                    if (this.getInstance()) {
                        this.getInstance().hide();
                    }

                    setTimeout(function() {
                        // MNSTR-1360, MNSTR-1356: we need setCaretToPosition primarily for IE11 and Edge in order to `focus` below works
                        this.$textarea.setCaretToPosition(0);
                        this.$textarea.focus();

                        result.resolve();
                    }.bind(this), 25);

                    this.$textarea.attr("originalValue", this.$textarea.val());
                } else {
                    var editorDirty = this.$textarea.attr("originalValue") !== this.$textarea.val();

                    this.showMask();
                    this.coverTextarea();

                    var renderMarkup = editorDirty ? controller.renderMarkup(this.$textarea.val(), this.getRenderParams()) : new $.Deferred().resolve();

                    renderMarkup.always(function() {
                        this.hideMask();

                        this.getInstance().show();
                        this.getInstance().focus();

                        if (editorDirty) {
                            renderMarkup.done(function (rendered) {
                                this.getInstance().setContent(rendered, null, true);

                                result.resolve();
                            }.bind(this)).fail(function () {
                                result.reject();
                                this.$textarea.removeAttr("originalValue");
                            }.bind(this));
                        } else {
                            result.resolve();
                        }
                    }.bind(this));
                }

                return result;
            },

            changeHandler: function(value) {
                controller.renderMarkup(value, this.getRenderParams()).done(function(rendered) {
                    this.getInstance().setContent(rendered);
                }.bind(this));
            },

            getInstance: function() {
                return this[PROPERTY];
            },

            loadingStart: function () {
                if (typeof this[PROPERTY] !== "undefined" || this[PROPERTY_LOADING]) {
                    return false;
                }

                this.undoManager = this.$textarea.data('undoManager');
                this[PROPERTY_LOADING] = true;
                this.coverTextarea();
                // tinymce loading is not instant, lets mitigate this problem
                this.showMask();

                return true;
            },

            setInstance: function (editorInstance) {
                this[PROPERTY] = editorInstance;
            },

            loadingStop: function () {
                delete this[PROPERTY_LOADING];
                this[PROPERTY].disable();
            },

            shouldFocus: function () {
                return this.$textarea.parents('#addcomment, #comment-add, .issue-body-content').length > 0;
            },

            /**
             * Write down current editor content to underlying textarea.
             * @returns {Deferred|*}
             */
            serialize: function() {
                var result = new $.Deferred();

                var content = this.getInstance().getContent();
                require('jira/editor/converter').convert(content).then(function (converted) {
                    logger.trace("jira.editor.plugin.converted");

                    this.$textarea
                        .attr("value", converted)
                        .trigger("input");

                    RoundTrip.verify(content, converted, this.getRenderParams());

                    result.resolve();
                }.bind(this));

                return result;
            }
        },

        attached: function(element) {
            if (!element.FSM) {
                element.FSM = new FSM();
            }

            if (document.body.contains(element)) {
                element.FSM.triggerEvent(FSM.Events.ATTACHED, element);
            }
        },

        detached: function(element) {
            if (!document.body.contains(element) && element.FSM) { // corner case for jQuery.wrap and skate issue(triggering this even though element is still in DOM)
                element.FSM.triggerEvent(FSM.Events.DETACHED, element);
            }
        }
    };

    var RichEditor = skate("rich-editor", {
        type: skate.types.TAG,

        prototype: skateHandler.prototype,

        attached: skateHandler.attached,
        detached: skateHandler.detached
    });

    return RichEditor;
});
