define('jira/editor/round-trip', [
    "jira/editor/analytics",
    "jira/editor/controller",
    "jira/editor/util/strings",
    "jira/data/local-storage",
    "jira/editor/util/edit-distance",
    "underscore",
    "jquery"
], function (
    Analytics,
    controller,
    Strings,
    LocalStorage,
    EditDistance,
    _,
    $
) {
    var SCORE_KEY = "jira.editor.score";
    var DEFAULT_SCORE = JSON.stringify({
        count: 0,
        average: 0.0,
        lastReport: 0
    });
    var COUNT_WRAP = 100; // clear average after some amount of time
    // those settings are defensive, we can change them in future once we are confident about this code
    var REPORT_DELAY = 1000 * 60 * 60 * 24; // report on daily basis
    var MIN_COUNT = COUNT_WRAP * 0.1;  // don't send the very first average
    var DEBOUNCE_DELAY = 30000;

    /**
     * Get stored round trip score
     * @returns {Object}
     */
    var get = function () {
        try {
            return JSON.parse(LocalStorage.getItem(SCORE_KEY) || DEFAULT_SCORE);
        } catch(e) {
            return JSON.parse(DEFAULT_SCORE);
        }
    };

    /**
     * Make it persistent between page views.
     * @param score
     * @returns {Boolean} if saving score was successful
     */
    var store = function (score) {
        try {
            LocalStorage.setItem(SCORE_KEY, JSON.stringify(score));
            return true;
        } catch(e) {
            return false;
        }
    };

    /**
     * Calculate new score, store and report.
     * @param value
     */
    var put = function (value) {
        var score = get();

        // https://en.wikipedia.org/wiki/Moving_average#Cumulative_moving_average
        score.average = (value + score.count * score.average) / (score.count + 1);
        score.count += 1;

        report(score);

        if (score.count > COUNT_WRAP) {
            score.count = 1;
            score.average = value;
        }

        store(score);
    };

    /**
     * Possibly report current score
     * @param score
     */
    var report = function (score) {
        if (Date.now() - score.lastReport < REPORT_DELAY || score.count < MIN_COUNT) {
            return;
        }

        score.lastReport = Date.now();

        if (store(score)) {
            Analytics.sendEvent("editor.instance.roundtrip.score", {
                average: score.average,
                count: score.count
            });
        }
    };

    /**
     * Return score(0..1) for given source and round-tripped HTML.
     * @param content
     * @param renderedContent
     * @param ts timestamp
     * @returns {number}
     */
    var calc = function(content, renderedContent, ts) {
        if (content === renderedContent) {
            return 1;
        }

        if (content.length === 0 || renderedContent.length === 0) {
            return 0;
        }

        var map = {
            _chars: 33 // start from ! character
        };
        var leftHash = hashNode(new DOMParser().parseFromString(content, "text/html").body, map);
        var rightHash = hashNode(new DOMParser().parseFromString(renderedContent, "text/html").body, map);

        var distance = EditDistance.getEditDistance(leftHash, rightHash);
        var score = 1 - (distance / Math.max(leftHash.length, rightHash.length));

        Analytics.sendEvent("editor.instance.roundtrip.score.raw", {
            distance: distance,
            score: score,
            sourceLength: content.length,
            targetLength: renderedContent.length,
            time: window.performance.now() - ts
        });

        return score;
    };

    /**
     * Given dom node, create hash representing each child node with a character
     *
     * @param node
     * @param map
     * @returns {string}
     */
    var hashNode = function(node, map) {
        var stack = _(node.childNodes).toArray();
        var hash = '';

        while(stack.length > 0) {
            var child = stack.splice(0, 1)[0];

            var nodeID = child.nodeName;
            if (child.nodeType === Node.TEXT_NODE) {
                nodeID += '#' + child.nodeValue;
            } else if (child.attributes.length > 0) {
                nodeID += '@' + _(child.attributes).toArray().filter(function(attr) {
                    return !/^(data-mce|style)/.test(attr.name);
                }).map(function(attr) {
                    return attr.name + '=' + attr.value;
                }).sort().join("@");
            }

            if (!map[nodeID]) {
                map[nodeID] = String.fromCharCode(map._chars++);
            }

            hash += map[nodeID];
            stack.push.apply(stack, child.childNodes);
        }

        return hash;
    };

    var normalize = function(content) {
        try {
            var result = '';
            var node = new DOMParser().parseFromString(content, "text/html").body;
            var normalizeNode = function(child) {
                return child.nodeType === Node.TEXT_NODE && /^\s+$/.test(child.nodeValue)
                    || child.nodeType === Node.COMMENT_NODE
                    || child.nodeName === "A" && !child.nodeValue;
            };

            $(node).find('>p:first-child:last-child').contents().unwrap();

            var stack = _(node.childNodes).toArray();

            while(stack.length > 0) {
                var child = stack.pop();

                if (normalizeNode(child)) {
                    child.parentNode.removeChild(child);
                    continue;
                } else  if (child.nodeName === "STRONG") {
                    var replacement = document.createElement('b');
                    _(child.childNodes).toArray().forEach(replacement.appendChild, replacement);
                    child.parentNode.replaceChild(replacement, child);
                    child = replacement;
                } else if (child.nodeName === "SPAN" && child.className === 'error') {
                    var text = $(child).text();
                    var filename = Strings.getFilenameFromError(text);
                    if (filename) {
                        $(child).replaceWith($('<img />').attr("data-filename", filename));
                    }
                } else if (child.nodeName === "IMG" && child.getAttribute("data-filename") && /^data:/.test(child.src)) {
                     child.removeAttribute("src");
                } else if (child.nodeName === "DIV" && child.className === "table-wrap") {
                    $(child).find("table:first-child:last-child").unwrap();
                }

                stack.push.apply(stack, child.childNodes);
            }

            return node.innerHTML;
        } catch(e) {
            return content;
        }
    };

    /**
     * Verify whether given markup will be rendered back to provided content and then send appropriate analytics event.
     *
     * This function is debounced, it should be triggered after some activity.
     *
     * @param content
     * @param markup
     * @param params
     */
    var verify = _.debounce(function (content, markup, params) {
        controller.renderMarkup(markup, params).done(function (renderedContent) {
            var ts = window.performance.now();
            put(calc(normalize(content), normalize(renderedContent), ts));
        }).fail(function () {
            // ignored
        });
    }, DEBOUNCE_DELAY);

    return {
        verify: verify,
        get: get
    }
});