/**
 * Provides a support for keyboard shortcuts for saving (submitting) the editor content.
 *
 * Two cases:
 * - a support for accesskey attribute (needed because of the iframe boundary)
 * - a support for Ctrl+Enter and Cmd+Enter (on Mac OS X)
 *
 * It's consistent with the Text mode (also the old Wiki Editor).
 * When doing changes verify the UX consistency and check the following:
 * - Forms.submitOnCtrlEnter in Jira Core
 * - com/atlassian/jira/util/BrowserUtils.java in Jira Core
 */
define("jira/editor/saving-kb-shortcuts", [
    "jquery",
    "jira/editor/analytics",
    "jira/util/navigator",
    "jira/util/data/meta",
    "jira/util/formatter"
], function (
    $,
    Analytics,
    Navigator,
    Meta,
    formatter
) {
    function isCtrlOrCmdEnterPressed(keydownEvent) {
        return (keydownEvent.ctrlKey || (Navigator.isMac() && keydownEvent.metaKey))
            && (keydownEvent.keyCode === 13 || (!Navigator.isMac() && (keydownEvent.keyCode === 10)));
    }

    function getAccessKeyModifiers() {
        // possible values can be found in com/atlassian/jira/util/BrowserUtils.java
        return (Meta.get("keyboard-accesskey-modifier") || '')
            .split('+').map(function(mod) {
                return mod.trim().toLowerCase() + 'Key';
            });
    }

    function getSubmitAccessKeyCode() {
        var submitAccessKey = formatter.I18n.getText('common.forms.submit.accesskey');
        return submitAccessKey && submitAccessKey.length === 1
            ? submitAccessKey.toUpperCase().charCodeAt(0) : 0;
    }

    var submitAccessKeyCode = getSubmitAccessKeyCode();
    var accessKeyModifiers = getAccessKeyModifiers();

    function isSubmitAccessKeyPressed(keydownEvent) {
        if (!submitAccessKeyCode || submitAccessKeyCode !== keydownEvent.keyCode) {
            return false;
        }

        return accessKeyModifiers.every(function(modKey) {
            return keydownEvent[modKey];
        });
    }

    function enable(richEditor) {
        richEditor.getInstance().relayEvent("keydown", function (e) {
            if (isCtrlOrCmdEnterPressed(e) || isSubmitAccessKeyPressed(e)) {
                e.preventDefault();

                richEditor.serialize().then(function () {
                    Analytics.sendEvent("editor.instance.submit");
                    $(richEditor.$textarea.prop("form")).submit();
                })
            }
        }, true);
    }

    return {
        enable: enable
    }
});
