define("jira/editor/analytics", [
    "jira/analytics",
    "jira/util/logger"
], function (
    Analytics,
    logger
) {
    var ANALYTIC_PERF_KEY = "editor.instance.perf.";

    var editorAnalytics = {};

    editorAnalytics.sendEvent = function (eventName, optionalData) {
        Analytics.send({
            name: eventName, properties: optionalData || {}
        });
    };

    /**
     * @see <a href="https://www.w3.org/TR/user-timing/#dom-performance-mark">spec</a>
     */
    editorAnalytics.mark = function(name) {
        window.performance.mark && window.performance.mark(name);
    }

    /**
     * @see <a href="https://www.w3.org/TR/user-timing/#dom-performance-measure">spec</a>
     */
    editorAnalytics.measure = function(measureName, markStart, markEnd) {
        window.performance.measure && window.performance.measure(measureName, markStart, markEnd);
    }

    /**
     * Does measure, clears mark entries and sends analytic event.
     *
     * You can use the same "markStart" to emit multiple events:
     * > oneShotMeasure(markStart, {}, firstEventName, true);
     * ...
     * > oneShotMeasure(markStart, {}, secondEventName, true);
     *
     * event_key = ANALYTIC_PERF_KEY + <markStart without "rte-" prefix, if present>
     * or if eventName is specified:
     * event_key = ANALYTIC_PERF_KEY + eventName
     *
     * @param {string} markStart the name of already stored mark, also used for event key
     * @param {Object=} extraAttributes those attributes will be sent in analytic event
     *                                  along with the 'time' attribute
     * @param {string=} eventName builds event key
     * @param {boolean=} keepEntries if true, then entries are not removed; default: false
     * @returns {Number} duration, elapsed time between 'markStart' and 'now';
     *                   negative value indicates error
     */
    editorAnalytics.oneShotMeasure = function(markStart, extraAttributes, eventName, keepEntries) {
        var api = window.performance;
        if (!api.mark) {
            logger.debug("The browser does not support User Timing API");
            return -1;
        }

        var endTime = api.now();

        var entries = api.getEntriesByName(markStart);
        var entry = entries.length && entries[entries.length - 1];
        if (!(entry instanceof window.PerformanceMark)) {
            logger.debug("The mark has not been found");
            return -1;
        }

        if (typeof eventName !== 'string') {
            eventName = markStart.replace(/^rte\-/, "");
        }

        var duration = doMeasureAndSendEvent(entry.startTime, endTime, extraAttributes, eventName);

        if (!keepEntries) {
            api.clearMarks(markStart);
        }

        return duration;
    }

    /**
     * @returns {{measure: Function}}
     */
    editorAnalytics.startMeasure = function() {
        var startTime = window.performance.now();
        return {
            /**
             * Does measure and sends analytic event based on the previously obtained `startTime`.
             * You can call it multiple times and the same `startTime` will be used.
             *
             * @param {string} eventName event_key = ANALYTIC_PERF_KEY + eventName
             * @param {object=} extraAttributes those attributes will be sent in analytic event
             *                                  along with the 'time' attribute
             * @returns {Number} duration, elapsed time between 'startTime' and 'now'
             */
            measure: function measure(eventName, extraAttributes) {
                return doMeasureAndSendEvent(startTime, window.performance.now(), extraAttributes, eventName);
            }
        };
    };

    return editorAnalytics;


    function doMeasureAndSendEvent(startTime, endTime, extraAttributes, eventName) {
        var duration = endTime - startTime;
        var eventKey = ANALYTIC_PERF_KEY + eventName;

        var attributes = extraAttributes || {};
        attributes.time = duration;

        editorAnalytics.sendEvent(eventKey, attributes);

        return duration;
    }
});