/**
 * Utility functions for parsing and manipulating string values.
 * @module jira/editor/converter/util/strings
 */
define('jira/editor/converter/util/strings', [
    'jira/editor/util/strings',
    'exports',
    'jquery',
    'underscore'
], function (Strings,
             exports,
             $,
             _) {
    'use strict';

    var EMOTE_PATH = '/images/icons/emoticons/';
    var ATTACHMENT_PATH = '/secure/attachment/';
    var THUMBNAIL_PATH = '/secure/thumbnail/';
    var TEMPORARY_ATTACHMENT_PATH = '/secure/temporaryattachment/';

    /**
     * Returns a boolean if the passed string is "true" or "false", ignoring case, else returns the original string.
     * @param {String} value
     * @returns {Boolean|String}
     * @since 5.0
     */
    exports.asBooleanOrString = function asBooleanOrString(value) {
        var lc = value ? value.toLowerCase() : '';
        var result = value;

        if (lc === 'true') {
            result = true;
        } else if (lc === 'false') {
            result = false;
        }

        return result;
    };

    /**
     * Looks up a search string from the beginning of the given string
     * @param str the given string
     * @param searchString search string
     * @returns {boolean} true if string starts with a search string.
     */
    exports.startsWith = function (str, searchString) {
        if (_.isEmpty(str)) {
            return false;
        }
        if (_.isEmpty(searchString)) {
            return false;
        }
        return str.indexOf(searchString) === 0;
    };

    /**
     * Check whether given string ends with a search string
     * @param str the given string
     * @param searchString search string
     * @returns {boolean} true if the given string ends with a search string
     */
    exports.endsWith = function (str, searchString) {
        if (_.isEmpty(str)) {
            return false;
        }
        if (_.isEmpty(searchString)) {
            return false;
        }
        return str.indexOf(searchString) + searchString.length === str.length;
    };

    /**
     * Looks up a search string in a given string, returns a substring of the given string after a full match.
     * @param str the given string
     * @param searchString search string
     * @returns {string} the substring of the given string after a full match. Returns empty string if the search string was not found.
     */
    exports.substringAfter = function (str, searchString) {
        if (_.isEmpty(str)) {
            return str;
        }
        if (_.isEmpty(searchString)) {
            return '';
        }
        var indexOf = str.indexOf(searchString);
        if (indexOf < 0) {
            return '';
        }
        return str.substring(indexOf + searchString.length);
    };

    /**
     * Looks up a search string in a given string, returns a substring of the given string before a full match.
     * @param str the given string
     * @param searchString search string
     * @returns {string} the substring of the given string before a full match. Returns empty string if the search string was not found.
     */
    exports.substringBefore = function (str, searchString) {
        if (_.isEmpty(str)) {
            return str;
        }
        if (_.isEmpty(searchString)) {
            return '';
        }
        var indexOf = str.indexOf(searchString);
        if (indexOf < 0) {
            return '';
        }
        return str.substring(0, indexOf);
    };

    /**
     * Looks up a search string in a given string.
     * @param str the given string
     * @param searchString search string
     * @returns {boolean} true, if substring exists in a given string.
     */
    exports.contains = function (str, searchString) {
        if (_.isEmpty(str)) {
            return false;
        }
        var indexOf = str.indexOf(searchString);
        return indexOf >= 0;
    };

    /**
     * Generates a new string by repeating the given string n times
     * @param str the given string
     * @param times times to repeat the given string
     * @returns {string} the generated string
     */
    exports.repeat = function (str, times) {
        if (_.isEmpty(str)) {
            return '';
        }
        return new Array(times + 1).join(str);
    };

    /**
     * Retuns the length of the string.
     * @param str string
     * @returns {int} length
     */
    exports.length = function (str) {
        if (_.isEmpty(str)) {
            return 0;
        }
        return str.length;
    };

    /**
     * Replaces each exact match of the searchString in a given string with a replaceString.
     * Does not support regular expressions.
     * @param str original string
     * @param searchString search string
     * @param replaceString replace string
     * @returns {string}
     */
    exports.replace = function (str, searchString, replaceString) {
        if (_.isEmpty(str)) {
            return ''
        }
        if (_.isEmpty(searchString)) {
            return ''
        }
        if (typeof replaceString === 'undefined') {
            replaceString = '';
        }

        var searchStart = 0;
        var i = str.indexOf(searchString);
        while (i >= 0) {
            var beforeOccurrence = str.substring(0, i);
            var afterOccurrence = str.substring(i + searchString.length);
            str = beforeOccurrence + replaceString + afterOccurrence;

            searchStart = i + replaceString.length;
            i = str.indexOf(searchString, searchStart);
        }
        return str.substring(i, str.length);
    };

    /**
     * Returns a hash code of a passed string by calculating
     * product of characters using 31 as multiplication factor.
     * It is good for anonymity, not suitable for security purposes.
     *
     * @param {String} toHash
     * @returns {String} hash code
     */
    exports.hashCode = function hashCode(toHash) {
        var charCode;
        var hash = 0;
        if (!toHash) {
            return "";
        }

        for (var i = 0; i < toHash.length; i += 1) {
            charCode = toHash.charCodeAt(i);
            hash = (hash * 31) + charCode;
            hash |= 0; // force 32-bit representation
        }

        return "" + hash;
    };

    exports.resolveUrl = function (url, base_url) {
        if (!url) {
            // no input means no output
            return;
        }

        var $base = $(document.createElement('base')).prop('href', base_url).appendTo('head');

        try {
            return $(document.createElement('a')).prop("href", url).prop("href")
        } finally {
            $base.remove();
        }
    };

    exports.getBaseUrl = function () {
        return document.location.origin + document.location.pathname;
    };

    exports.getContextUrl = function () {
        return document.location.origin + AJS.contextPath();
    };

    exports.getFilenameFromHref = function (href, temporaryAttachment) {
        var hrefArray = href.split('/');
        var filename = hrefArray[hrefArray.length - 1];

        if (!temporaryAttachment) {
            //remove id from the filename
            var id = hrefArray[hrefArray.length - 2];
            var start = id + "_";
            if (filename.indexOf(start) === 0) {
                filename = filename.substring(start.length);
            }
        } else {
            //remove id from the filename
            var idx = filename.indexOf("_");
            if (idx > 0) {
                filename = filename.substring(idx + 1);
            }
        }

        filename = filename.replace(new RegExp('\\+', 'g'), ' ');
        return decodeURIComponent(filename);
    };

    exports.isAttachmentPath = function (href) {
        return exports.contains(href, ATTACHMENT_PATH);
    };

    exports.isTemporaryAttachmentPath = function (href) {
        return exports.contains(href, TEMPORARY_ATTACHMENT_PATH);
    };

    exports.isThumbnailPath = function (href) {
        return exports.contains(href, THUMBNAIL_PATH);
    };

    exports.isEmotePath = function (href) {
        return exports.contains(href, EMOTE_PATH);
    };

    exports.getFilenameFromError = function (text) {
        return Strings.getFilenameFromError(text);
    };
});
