define("jira/editor/converter/handlers/link-handler", [
    "jira/editor/converter/handlers/node-handler",
    "jira/editor/converter/handlers/text-handler",
    "jira/editor/converter/handlers/image-handler",
    "jira/editor/converter/handlers/empty-handler",
    "jira/editor/converter/handlers/superscript-handler",
    "jira/editor/converter/util/strings",
    "jira/editor/converter/util/uri",
    "jquery"
], function (
    NodeHandler,
    TextHandler,
    ImageHandler,
    EmptyHandler,
    SuperscriptHandler,
    Strings,
    uri,
    $
) {
    "use strict";

    /**
     * @class LinkHandler
     * @extends NodeHandler
     */
    var LinkHandler = NodeHandler.extend({
        init: function (node) {
            this.node = node;

            var attrHref = this.node.getAttribute("href");
            var href = Strings.resolveUrl(attrHref, Strings.getBaseUrl());
            var filename = node.getAttribute('data-filename');

            if (LinkHandler.isAnchor(attrHref)) {
                href = attrHref;
            } else if (LinkHandler.isAttachment(href) || filename) {
                // href is an issue link: (context path)/browse/KEY-1000 -> KEY-1000
                this.attachment = filename || Strings.getFilenameFromHref(href);
                if ($(node).find('>:first-child:last-child').is('img')) {
                    this.imageAttachment = this.attachment;
                }
            } else if (LinkHandler.isIssue(href)) {
                // href is an issue link: (context path)/browse/KEY-1000 -> KEY-1000
                this.issueKey = Strings.substringAfter(href, Strings.getContextUrl() + '/browse/');
            } else if (LinkHandler.isProfile(href, node)) {
                // src is mentioning link: (context path)/secure/ViewProfile?jspa?name=username -> [~username]
                this.username = uri.decodeUri(Strings.substringAfter(href, 'ViewProfile.jspa?name='));
            } else if (LinkHandler.isWiki(href, node)) {
                // src is mentioning link: /wiki/display/PROJECTKEY/PAGEKEY -> [PAGEKEY]
                this.wikiLink = $(node).text();
            }

            this.href = href;
        },

        before: function () {
            var whitespace = $(this.node.previousSibling).is('a') ? ' ' : '';

            if (this.imageAttachment) {
                return TextHandler.createTextHandler(whitespace + '!');
            } else if (this.attachment) {
                return TextHandler.createTextHandler(whitespace + '[^');
            } else if (this.issueKey) {
                // issue key is shown as plain text, nothing to do
                return new EmptyHandler();
            } else if (this.username) {
                return TextHandler.createTextHandler(whitespace + '[~');
            } else if (this.href || this.wikiLink) {
                return TextHandler.createTextHandler(whitespace + '[');
            } else {
                return new EmptyHandler();
            }
        },

        children: function () {
            // RTE-5: Have to stop children processing for these cases:
            // * attachments
            // * username
            // as the link text got parsed as child, not the url attribute we parse
            if (this.imageAttachment) {
                return [TextHandler.createTextHandler('' + this.imageAttachment)];
            } else if (this.attachment) {
                return [TextHandler.createTextHandler('' + this.attachment)];
            } else if (this.username) {
                return [TextHandler.createTextHandler('' + this.username)]
            } else if (LinkHandler.isAnchor(this.href, this.node)) {
                return TextHandler.createTextHandler(this.href);
            } else if (LinkHandler.isMailto(this.href, this.node)) {
                return this._filterMailtoChildren(this._super());
            } else {
                return this._super();
            }
        },

        after: function () {
            if (this.imageAttachment) {
                var attrs = ImageHandler.getAttrs($(this.node).find('> img')[0]);
                return TextHandler.createTextHandler(attrs + '!');
            } else if (this.attachment || this.wikiLink) {
                return TextHandler.createTextHandler(']');
            } else if (this.issueKey) {
                // issue key is shown as plain text, nothing to do
                return new EmptyHandler();
            } else if (this.username) {
                return TextHandler.createTextHandler(']');
            } else if (this.href) {
                if (this.href === $(this.node).text() || LinkHandler.isAnchor(this.href, this.node)) {
                    return TextHandler.createTextHandler(']');
                } else {
                    return TextHandler.createTextHandler('|' + this.href + ']');
                }
            } else {
                return new EmptyHandler();
            }
        },

        // JIRA automatically adds image in <sup> tag for mailto link, but we don't want to process it.
        // If the last chilren in mailto link is superscript, than we ignore it(pop-ing) from children array.
        // We pop last 3 elements from array, 'cause sup-node is wrapped by empty-handlers.
        _filterMailtoChildren: function (children) {
            if (children.length >= 3 && (children[children.length - 2] instanceof SuperscriptHandler)) {
                children = children.splice(0, children.length - 3);
            }
            return children;
        }
    });

    LinkHandler.isAttachment = function (href) {
        if(/^\//.test(href)) {
            href = window.location.origin + href;
        }
        return Strings.startsWith(href, Strings.getContextUrl() + '/secure/attachment/') || Strings.startsWith(href, '../secure/attachment/');
    };

    LinkHandler.isIssue = function (href) {
        if(/^\//.test(href)) {
            href = window.location.origin + href;
        }
        return Strings.startsWith(href, Strings.getContextUrl() + '/browse/');
    };

    LinkHandler.isProfile = function (href, node) {
        return Strings.contains(href, 'ViewProfile.jspa?name=') && $(node).is(".user-hover");
    };

    LinkHandler.isWiki = function (href, node) {
        return Strings.contains(href, '/wiki/display/') && Strings.endsWith(uri.decodeUri(href), '/' + $(node).text());
    };

    LinkHandler.isMailto = function (href, node) {
        return Strings.startsWith(href, 'mailto:');
    };

    LinkHandler.isAnchor = function (href, node) {
        return Strings.startsWith(href, '#') && (!node || href === ('#' + $(node).text()));
    };

    return LinkHandler;
});
