define("jira/editor/converter/handlers/block-handler", [
    "jira/editor/converter/handlers/node-handler",
    "jira/editor/converter/handlers/text-handler",
    "jira/editor/converter/new-lines/new-lines",
    "jira/editor/converter/util/strings",
    "jira/editor/converter/util/color-converter",
    "jquery"
], function (
    NodeHandler,
    TextHandler,
    NewLines,
    Strings,
    ColorConverter,
    $
) {
    "use strict";

    var BlockHandler = NodeHandler.extend({

        init: function BlockHandler(node) {
            this.wrapperNode = node;
            this.node = node.querySelector('.panelContent') || node;
            this.nodeHeader = node.querySelector('panel-title') || node.querySelector('.panelHeader');
            this.attrs = this.getAttrs();

            if ($(this.node).is(".panelContent")) {
                this.workaroundTypingOutsideWrapper();
            }
        },

        workaroundTypingOutsideWrapper: function () {
            var $node = $(this.wrapperNode);

            // Add line breaks prior to concatenating paragraphs as preformatted text
            $node.find('> p').before('<br data-mce-bogus="1">');

            // Move content that might be typed (by exploiting SHIFT+ENTER) outside of wrapper <div><pre> into panelContent <div>
            if (this.node) {
                var node = $node.find('> :not(div.preformattedContent.panelContent, div.codeContent.panelContent, div.panelHeader, div.panelContent)');
                node.detach().appendTo(this.node);
            } else {
                this.node = this.wrapperNode;
            }
        },

        getAttrs: function () {
            var attrs = [];

            if (this.nodeHeader) {
                if (this.nodeHeader.textContent) {
                    attrs.push('title=' + this.nodeHeader.textContent);
                }
                if (this.nodeHeader.style.borderBottomStyle) {
                    attrs.push('borderStyle=' + this.nodeHeader.style.borderBottomStyle);
                }
                if (this.nodeHeader.style.borderBottomWidth && this.nodeHeader.style.borderBottomWidth !== "1px") {
                    attrs.push('borderWidth=' + this.nodeHeader.style.borderBottomWidth);
                }
                if (this.nodeHeader.style.borderBottomColor) {
                    attrs.push('borderColor=' + ColorConverter.colorToHex(this.nodeHeader.style.borderBottomColor));
                }
                if (this.nodeHeader.style.backgroundColor) {
                    attrs.push('titleBGColor=' + ColorConverter.colorToHex(this.nodeHeader.style.backgroundColor));
                }
            }
            if (this.wrapperNode) {
                if (this.wrapperNode.style.backgroundColor) {
                    attrs.push('bgColor=' + ColorConverter.colorToHex(this.wrapperNode.style.backgroundColor));
                }
            }

            return attrs.length > 0 ? attrs.join('|') : '';
        },

        createChildHandler: function (node, childIndex, childrenTotal) {
            if (node.nodeType === Node.TEXT_NODE) {
                return TextHandler.createPreformattedTextHandler(node);
            } else if (node.tagName === 'BR') {
                return TextHandler.newlineHandler();
            } else if (node.tagName === 'IMG') {
                if (!Strings.startsWith(node.src, "data") && !Strings.startsWith(node.src, "blob")) {
                    return NodeHandler.createHandler(node);
                } else {
                    var fakeNode = document.createTextNode("(attachment)");
                    return TextHandler.createTextHandler(fakeNode);
                }
            } else if (node.tagName === 'PRE') {
                return new BlockInternalNodeHandler(node);
            } else if (node.tagName === 'PANEL-TITLE') {
                return new NodeHandler.EmptyHandler(node);
            } else {
                return new BlockInternalNodeHandler(node);
            }
        },

        children: function() {
            var children = NodeHandler.prototype.children.apply(this, arguments);
            children.unshift(NewLines.single(true));
            return children;
        },

        newLinesBefore: function () {
            return NodeHandler.singleNewLineExceptTable(this.wrapperNode, true);
        },

        newLinesAfter: function () {
            return NodeHandler.singleNewLineExceptTable(this.wrapperNode, true);
        }
    });

    var BlockInternalNodeHandler = NodeHandler.extend({
        createChildHandler: BlockHandler.prototype.createChildHandler,

        newLinesBefore: function () {
            return NewLines.empty();
        },

        before: function () {
            return new NodeHandler.EmptyHandler();
        },

        after: function () {
            return new NodeHandler.EmptyHandler();
        }
    });

    return BlockHandler;
});