AJS.test.require([
    'com.atlassian.jira.plugins.jira-editor-plugin:api',
    'com.atlassian.jira.plugins.jira-editor-plugin:marionette'
], function () {

    var $ = require('jquery');
    var _ = require("underscore");
    var Backbone = require("backbone");

    /**
     * Call it with the module context.
     */
    function setup() {
        this.context = AJS.test.mockableModuleContext();
        this.sandbox = sinon.sandbox.create();

        this.editor = _.extend({
            pasteInsidePreSchemaSpec: {},
            pasteSchemaSpec: {},
            editor: {
                selection: {
                    getNode: function () {
                        return this.selectedNode;
                    }.bind(this)
                }
            }
        }, Backbone.Events);

        this.context.mock('jira/editor/context-detector', {
            detectPre: sinon.stub()
        });
        this.contextDetector = this.context.require('jira/editor/context-detector');

        var ContextManager = this.context.require('jira/editor/context-manager');
        this.contextManager  = new ContextManager(this.editor);
        this.sandbox.spy(this.contextManager, "trigger");

        this.changedAllSpy = this.contextManager.trigger.withArgs("change:all");
        this.changedTableSpy = this.contextManager.trigger.withArgs("change:table");
        this.changedASpy = this.contextManager.trigger.withArgs("change:a");
    }

    module("ContextManager", {
        setup: function() {
            setup.call(this);
        },

        teardown: function () {
            this.sandbox.restore();
        }
    });


    /* common */
    test('Should not trigger the same event twice', function () {
        function triggerUpdate (editor) {
            editor.trigger("selection:update", {
                insidePreformatted: true,
                preformattedSelected: false,
                insideTable: false
            });
        }

        triggerUpdate(this.editor);
        triggerUpdate(this.editor);

        sinon.assert.calledOnce(this.changedAllSpy);
        sinon.assert.calledOnce(this.changedTableSpy);
    });

    test('Should transition from "null" states to "false" states (initialisation)', function () {
        this.editor.trigger("selection:update", {
            insidePreformatted: false,
            preformattedSelected: false,
            insideTable: false
        });

        sinon.assert.calledOnce(this.changedAllSpy);
        sinon.assert.calledOnce(this.changedTableSpy);
    });

    /* "all" state */
    test('Trigger disable event "all" when caret is inside pre', function() {
        this.editor.trigger("selection:update", {
            insidePreformatted: true,
            preformattedSelected: false
        });

        sinon.assert.calledWith(this.contextManager.trigger, "change:all", {
            disableState: true
        });
    });

    test('Trigger disable event "all" when pre is selected', function() {
        this.editor.trigger("selection:update", {
            insidePreformatted: false,
            preformattedSelected: true
        });

        sinon.assert.calledWith(this.contextManager.trigger, "change:all", {
            disableState: true
        });
    });

    test('Trigger enable event "all" when there is no pre', function() {
        this.editor.trigger("selection:update", {
            insidePreformatted: false,
            preformattedSelected: false
        });

        sinon.assert.calledWith(this.contextManager.trigger, "change:all", {
            disableState: false
        });
    });

    test('Trigger enable event "all" when change active tab', function() {
        this.editor.trigger("tabs:changed");

        sinon.assert.calledWith(this.contextManager.trigger, "change:all", {
            disableState: false
        });
    });

    /* "table" state */
    test('Trigger disable event "table" when caret is inside a table', function() {
        this.editor.trigger("selection:update", {
            insideTable: true
        });

        sinon.assert.calledWith(this.changedTableSpy, sinon.match.string, {
            disableState: true
        });
    });

    test('Trigger enable event "table" when caret is outside a table', function() {
        this.editor.trigger("selection:update", {
            insideTable: false
        });

        sinon.assert.calledWith(this.changedTableSpy, sinon.match.string, {
            disableState: false
        });
    });

    /* "a" state */
    test('Trigger disable event "a" when caret is inside A', function() {
        this.editor.trigger("selection:update", {
            insideA: true
        });

        sinon.assert.calledWith(this.changedASpy, sinon.match.string, {
            disableState: true
        });
    });

    test('Trigger enable event "a" when caret is outside A', function() {
        this.editor.trigger("selection:update", {
            insideA: false
        });

        sinon.assert.calledWith(this.changedASpy, sinon.match.string, {
            disableState: false
        });
    });

    /* paste schema */
    test('Should use normal PasteSchema outside "pre" context', function() {
        this.contextDetector.detectPre.returns(false);
        equal(this.contextManager.getPasteSchemaForContext(), this.editor.pasteSchemaSpec);
    });

    test('PasteSchema should be pre if inside code', function() {
        this.contextDetector.detectPre.returns(true);
        equal(this.contextManager.getPasteSchemaForContext(), this.editor.pasteInsidePreSchemaSpec);
    });


    /* state transitions */
    module('ContextManager: "table" state transitions #1 (from no context)', {
        setup: function() {
            setup.call(this);

            this.editor.trigger("selection:update", {
                insidePreformatted: false,
                preformattedSelected: false,
                insideTable: false
            });

            this.contextManager.trigger.reset();
        },

        teardown: function () {
            this.sandbox.restore();
        }
    });

    test('Should transition from no context to "pre" context properly', function() {
        this.editor.trigger("selection:update", {
            insidePreformatted: true,
            preformattedSelected: true,
            insideTable: false
        });

        sinon.assert.calledOnce(this.changedAllSpy);
        sinon.assert.calledWith(this.changedAllSpy, sinon.match.string, {
            disableState: true
        });

        sinon.assert.calledOnce(this.changedTableSpy);
        sinon.assert.calledWith(this.changedTableSpy, sinon.match.string, {
            disableState: true
        });
    });

    test('Should transition from no context to "table" context properly', function() {
        this.editor.trigger("selection:update", {
            insidePreformatted: false,
            preformattedSelected: false,
            insideTable: true
        });

        sinon.assert.notCalled(this.changedAllSpy);

        sinon.assert.calledOnce(this.changedTableSpy);
        sinon.assert.calledWith(this.changedTableSpy, sinon.match.string, {
            disableState: true
        });
    });

    test('Should transition from no context to all contexts properly', function() {
        this.editor.trigger("selection:update", {
            insidePreformatted: true,
            preformattedSelected: true,
            insideTable: true
        });

        sinon.assert.calledOnce(this.changedAllSpy);
        sinon.assert.calledWith(this.changedAllSpy, sinon.match.string, {
            disableState: true
        });

        sinon.assert.calledOnce(this.changedTableSpy);
        sinon.assert.calledWith(this.changedTableSpy, sinon.match.string, {
            disableState: true
        });
    });


    module('ContextManager: "table" state transitions #2 (from "table" context)', {
        setup: function() {
            setup.call(this);

            this.editor.trigger("selection:update", {
                insidePreformatted: false,
                preformattedSelected: false,
                insideTable: true
            });

            this.contextManager.trigger.reset();
        },

        teardown: function () {
            this.sandbox.restore();
        }
    });

    test('Should transition from "table" context to no context properly', function() {
        this.editor.trigger("selection:update", {
            insidePreformatted: false,
            preformattedSelected: false,
            insideTable: false
        });

        sinon.assert.notCalled(this.changedAllSpy);

        sinon.assert.calledOnce(this.changedTableSpy);
        sinon.assert.calledWith(this.changedTableSpy, sinon.match.string, {
            disableState: false
        });
    });

    test('Should transition from "table" context to "pre" context properly', function() {
        this.editor.trigger("selection:update", {
            insidePreformatted: true,
            preformattedSelected: true,
            insideTable: false
        });

        sinon.assert.calledOnce(this.changedAllSpy);
        sinon.assert.calledWith(this.changedAllSpy, sinon.match.string, {
            disableState: true
        });

        sinon.assert.notCalled(this.changedTableSpy);
    });

    test('Should transition from "table" context to all contexts properly', function() {
        this.editor.trigger("selection:update", {
            insidePreformatted: true,
            preformattedSelected: true,
            insideTable: true
        });

        sinon.assert.calledOnce(this.changedAllSpy);
        sinon.assert.calledWith(this.changedAllSpy, sinon.match.string, {
            disableState: true
        });

        sinon.assert.notCalled(this.changedTableSpy);
    });


    module('ContextManager: "table" state transitions #3 (from "pre" context)', {
        setup: function() {
            setup.call(this);

            this.editor.trigger("selection:update", {
                insidePreformatted: true,
                preformattedSelected: true,
                insideTable: false
            });

            this.contextManager.trigger.reset();
        },

        teardown: function () {
            this.sandbox.restore();
        }
    });

    test('Should transition from "pre" context to no context properly', function() {
        this.editor.trigger("selection:update", {
            insidePreformatted: false,
            preformattedSelected: false,
            insideTable: false
        });

        sinon.assert.calledOnce(this.changedAllSpy);
        sinon.assert.calledWith(this.changedAllSpy, sinon.match.string, {
            disableState: false
        });

        sinon.assert.calledOnce(this.changedTableSpy);
        sinon.assert.calledWith(this.changedTableSpy, sinon.match.string, {
            disableState: false
        });
    });

    test('Should transition from "pre" context to "table" context properly', function() {
        this.editor.trigger("selection:update", {
            insidePreformatted: false,
            preformattedSelected: false,
            insideTable: true
        });

        sinon.assert.calledOnce(this.changedAllSpy);
        sinon.assert.calledWith(this.changedAllSpy, sinon.match.string, {
            disableState: false
        });

        ok(this.changedTableSpy.lastCall.args[1].disableState === true, '"table" context should be active');

        sinon.assert.callOrder(this.changedAllSpy, this.changedTableSpy);
    });

    test('Should transition from "pre" context to all contexts properly', function() {
        this.editor.trigger("selection:update", {
            insidePreformatted: true,
            preformattedSelected: true,
            insideTable: true
        });

        sinon.assert.notCalled(this.changedAllSpy);
        sinon.assert.notCalled(this.changedTableSpy);
    });


    module('ContextManager: "table" state transitions #4 (from all contexts)', {
        setup: function() {
            setup.call(this);

            this.editor.trigger("selection:update", {
                insidePreformatted: true,
                preformattedSelected: true,
                insideTable: true
            });

            this.contextManager.trigger.reset();
        },

        teardown: function () {
            this.sandbox.restore();
        }
    });

    test('Should transition from all contexts to no context properly', function() {
        this.editor.trigger("selection:update", {
            insidePreformatted: false,
            preformattedSelected: false,
            insideTable: false
        });

        sinon.assert.calledOnce(this.changedAllSpy);
        sinon.assert.calledWith(this.changedAllSpy, sinon.match.string, {
            disableState: false
        });

        sinon.assert.calledOnce(this.changedTableSpy);
        sinon.assert.calledWith(this.changedTableSpy, sinon.match.string, {
            disableState: false
        });
    });

    test('Should transition from all contexts to "pre" context properly', function() {
        this.editor.trigger("selection:update", {
            insidePreformatted: true,
            preformattedSelected: true,
            insideTable: false
        });

        sinon.assert.notCalled(this.changedAllSpy);
        sinon.assert.notCalled(this.changedTableSpy);
    });

    test('Should transition from all contexts to "table" context properly', function() {
        this.editor.trigger("selection:update", {
            insidePreformatted: false,
            preformattedSelected: false,
            insideTable: true
        });

        sinon.assert.calledOnce(this.changedAllSpy);
        sinon.assert.calledWith(this.changedAllSpy, sinon.match.string, {
            disableState: false
        });

        ok(this.changedTableSpy.lastCall.args[1].disableState === true, '"table" context should be active');

        sinon.assert.callOrder(this.changedAllSpy, this.changedTableSpy);
    });
});