define('jira/editor/registry', [
    'underscore',
    'backbone',
    'jira/editor/constants',
    'jira/editor/registry/entry'
], function (
    _,
    Backbone,
    Constants,
    Entry
) {

    var Modes = Constants.Modes;

    /**
     * <p>
     *     Registry keeping entries about all and every RTE editor on the page.
     *     Primary usage is as an event bus for editor registration/unregistration/mode switch events.
     *     Has {Backbone.Events} mixed into it, so basically reference to its' API for details.
     * </p>
     *
     * Possible event names:
     * <ul>
     *     <li><pre>register</pre></li>
     *     <li><pre>unregister</pre></li>
     *     <li><pre>switch</pre></li>
     * </ul>
     *
     * Every time "event" happens 4 different events are fired, namespaced as follows:
     * <ul>
     *     <li><pre>eventName</pre></li>
     *     <li><pre>eventName:mode</pre></li>
     *     <li><pre>eventName:mode:id</pre></li>
     *     <li><pre>eventName::id</pre></li>
     * </ul>
     *
     * E.g.: when showing up "Add comment" form with Text mode being default:
     * <pre>    'register', 'register:text', 'register:text:comment', 'register::comment'</pre>
     * then, when switching to Visual mode:
     * <pre>    'switch', 'switch:wysiwyg', 'switch:wysiwyg:comment', 'switch::comment'</pre>
     *
     * @see Backbone.Events
     * @class
     * @mixes Backbone.Events
     */
    function EditorRegistry() {
        this._instances = {};
    }

    _.extend(EditorRegistry.prototype, Backbone.Events, {
        /**
         * Retrieves entry by its' id
         *
         * @param id
         * @returns {Entry}
         */
        getEntry: function (id) {
            return this._instances[id];
        },

        _registerTextMode: function (textArea) {
            if (!textArea) {
                return;
            }

            var id = this._getTextAreaId(textArea);
            var entry = new Entry(id, textArea, this);

            this._register(entry, Modes.TEXT);
        },

        _registerVisualMode: function (rteInstance) {
            if (!rteInstance) {
                return;
            }

            var textArea = this._findTextArea(rteInstance);
            var id = this._getTextAreaId(textArea);

            var entry = this.getEntry(id);
            var isTextModeInitialized = !!entry;

            if (isTextModeInitialized) {
                this._resolveRTEInstance(entry, rteInstance);
                this._switchMode(textArea, Modes.VISUAL);
            } else {
                entry = new Entry(id, textArea, this);
                this._resolveRTEInstance(entry, rteInstance);
                this._register(entry, Modes.VISUAL);
            }
        },

        _findTextArea: function (rteInstance) {
            var richEditor = rteInstance.element;
            var wikiEditContent = richEditor.getContainer().parent().get(0);
            return wikiEditContent.querySelector('textarea:last-of-type');
        },

        _getTextAreaId: function (textArea) {
            var editorId = textArea.getAttribute('data-editor-id');
            if (!!editorId) {
                return editorId;
            }

            editorId = _.uniqueId(new Date().getTime() + '-');
            textArea.setAttribute('data-editor-id', editorId);
            return editorId;
        },

        _register: function (entry, mode) {
            entry.currentMode = mode;
            this._instances[entry.id] = entry;

            this._triggerEvents('register', entry);
        },

        _resolveRTEInstance: function (entry, rteInstance) {
            if (entry._rteInstance.state() === "resolved" || !rteInstance) {
                return;
            }

            entry._rteInstance.resolve(rteInstance);
        },

        _switchMode: function (textArea, mode) {
            var id = this._getTextAreaId(textArea);
            var entry = this.getEntry(id);
            if (!entry || entry.currentMode === mode) {
                return;
            }

            entry.currentMode = mode;

            this._triggerEvents('switch', entry);
        },

        _unregister: function (textArea) {
            var id = this._getTextAreaId(textArea);
            var entry = this.getEntry(id);
            if (!entry) {
                return;
            }

            delete this._instances[id];

            this._triggerEvents('unregister', entry);
        },

        _triggerEvents: function (eventName, entry) {
            this.trigger(eventName, entry);
            this.trigger(eventName + ':' + entry.currentMode, entry);
            this.trigger(eventName + ':' + entry.currentMode + ':' + entry.id, entry);
            this.trigger(eventName + '::' + entry.id, entry);
        }
    });

    return new EditorRegistry();
});