Backbone.define("JIRA.DevStatus.DetailDialogReviewView", JIRA.DevStatus.BaseDetailDialogView.extend({
    template: JIRA.Templates.DevStatus.DetailDialog.review,

    events: {
        'click a.review-link': '_onClickReviewLink',
        'click a.comment-link': '_onClickReviewLink'
    },

    initialize: function(options) {
        this.analytics = JIRA.DevStatus.ReviewsAnalytics;
        this.finalStates = ['REJECTED', 'CLOSED'];
        this.nonFinalStates = ['REVIEW', 'APPROVAL', 'SUMMARIZE'];
        this.validStates = _.union(this.finalStates, this.nonFinalStates);

        JIRA.DevStatus.BaseDetailDialogView.prototype.initialize.call(this, _.extend({
            type: "review",
            width: 1000,
            height: 400
        }, options));
    },

    /**
     * Sort reviewers by completion status, add moderator as reviewer, remove author from reviewers list
     */
    _updateReviewers: function(review) {
        review.reviewers = _
            .chain(review.reviewers)
            .filter(function (reviewer) {
                if (reviewer && review.author && review.author.name && reviewer.name) {
                    return !_.isEqual(reviewer.name, review.author.name);
                }
                return true;
            })
            .sortBy(function (reviewer) {
                return (reviewer.complete ? '0' : '1') + reviewer.name;
            }).map(function (reviewer) {
                reviewer.approved = reviewer.complete;
                return reviewer;
            }).value();

        if (review.moderator && !_.isEqual(review.moderator, review.author)) {
            review.reviewers = [_.extend(review.moderator, {name: AJS.I18n.getText('devstatus.detail.panel.review.tooltip.moderator') +
                ' ' + review.moderator.name})].concat(review.reviewers);
        }

        return review;
    },

    /**
     * Check for overdue reviews and mark them, strip due dates from final states
     */
    _updateReview: function(finalStates) {
        return function(review) {
            if (review.state && review.dueDate && _.contains(finalStates, review.state)) {
                delete review.dueDate;
            }

            if (review.dueDate && moment(review.dueDate).isBefore(moment())) {
                review.overDue = true;
            }

            return review;
        }
    },

    _addExtraReviewersJson: function(reviewersThreshold) {
        return function(review) {
            if (review.reviewers) {
                review.extraReviewers = JSON.stringify(review.reviewers.slice(reviewersThreshold));
            }
            return review;
        };
    },

    _reviewCompare: function(left, right) {
        if (left.dueDate && !right.dueDate) {
            return -1;
        } else if (!left.dueDate && right.dueDate) {
            return 1;
        } else if (left.dueDate && right.dueDate) {
            // earliest due date first
            return left.dueDate.localeCompare(right.dueDate);
        } else if (left.lastModified && right.lastModified) {
            // latest last updated first
            return right.lastModified.localeCompare(left.lastModified);
        }

        return 0;
    },

    /** Merge data from separate instances */
    _mergeInstanceData: function(memo, data) {
        return memo.concat(data.reviews || []);
    },

    /**
     * Transforms the input data, preparing it for disiplay. Filters invalid review states, sort reviews and reviewers.
     */
    _prepareReviews: function(reviewData) {
        var self = this;

        return _.chain(reviewData)
            .reduce(this._mergeInstanceData, [])
            .filter(function (review) {
                // State should _always_ be uppercase
                review.state = review.state && review.state.toUpperCase();
                // filter invalid states
                return _.contains(self.validStates, review.state);
            })
            .map(
                _.compose(this._addExtraReviewersJson(this.options.reviewersThreshold),
                    this._updateReviewers,
                    this._updateReview(this.finalStates)
                )
            )
            .sort(this._reviewCompare)
            .value();
    },

    renderSuccess: function(applicationType, result) {
        var reviews = this._prepareReviews(result),
            container = this.getContentContainer(applicationType);

        if (reviews.length > 0) {
            container.html(this.template({
                applicationType: applicationType,
                reviews: reviews,
                livestampRelativizeThreshold: this.options.livestampRelativizeThreshold,
                reviewersThreshold: this.options.reviewersThreshold
            }));

            this._createReviewersInlineDialog();
            JIRA.DevStatus.Date.addTooltip(container);
        } else {
            container.empty();
        }

        this.renderNoPermissionToViewAllWarningAtBottom(applicationType, reviews.length);

        return this;
    },

    getTitle: function(count, issue) {
        return AJS.I18n.getText("devstatus.detail.panel.title.review", count, issue);
    },

    getNoPermissionToViewAllMessageInCanvas: function() {
        return AJS.I18n.getText('devstatus.authentication.no-permission-to-view-all.review.title');
    },

    getOAuthMessageInCanvas: function() {
        return AJS.I18n.getText('devstatus.authentication.authenticate.review.title');
    },

    getOAuthMessageInFooter: function(instances) {
        return AJS.I18n.getText('devstatus.authentication.message.review', instances.length);
    },

    getConnectionMessageInCanvas: function() {
        return AJS.I18n.getText('devstatus.authentication.connection-problem.review.title');
    },

    hasData: function(detail) {
        return _.flatten(_.pluck(detail, "reviews"), true).length > 0;
    },

    _onClickReviewLink: function() {
        this.analytics.fireDetailReviewClicked();
    }
}));