AJS.test.require("com.atlassian.jira.plugins.jira-development-integration-plugin:devstatus-pullrequest-panel-resources");
AJS.test.require("com.atlassian.jira.plugins.jira-development-integration-plugin:devstatus-panel-resources");
AJS.test.require("com.atlassian.jira.plugins.jira-development-integration-plugin:devstatus-contract-test-resource");
AJS.test.require("com.atlassian.jira.plugins.jira-development-integration-plugin:devstatus-qunit-test-utils");

module("JIRA.DevStatus.DetailDialogPullRequestView", {
    setup: function() {
        this.sandbox = sinon.sandbox.create();
        this.issueKey = "DEV-1";
        this.issueId = 10000;
        this.server = sinon.fakeServer.create();
        JIRA.DevStatus.QunitTestUtils.spyAnalyticMethods("PullRequests", this.sandbox);
    },
    teardown : function () {
        this.sandbox.restore();
        this.server.restore();
        JIRA.DevStatus.QunitTestUtils.cleanAllDialogs();
        //JIRA form dialog appends overflow: hidden to the body of the page. This is to remove it.
        AJS.$("body").css("overflow", "initial");
    },
    getDetailDialog: function() {
        var dialog = AJS.$(".jira-dialog");
        return {
            heading: dialog.find(".devstatus-dialog-heading"),
            content: dialog.find(".devstatus-dialog-content"),

            getPullRequestContainer: function() {
                return this.content.find(".detail-pullrequests-container");
            },
            getRows: function() {
                return this.getPullRequestContainer().find(".pullrequest-row");
            },
            getColumns: function(row) {
                var $row = AJS.$(row);
                return {
                    author: $row.find(".author"),
                    state: $row.find(".state"),
                    id: $row.find(".pullrequest-id"),
                    title: $row.find(".title"),
                    commentCount: $row.find(".comment"),
                    reviewer: $row.find(".reviewer"),
                    timestamp: $row.find(".last-updated")
                }
            },
            getLinks: function(row) {
                var columns = this.getColumns(row);
                return {
                    pullRequest: columns.title.find(".pullrequest-link")
                }
            },
            getNoPermissionToViewAllMessage: function() {
                return this.content.find(".no-permission-to-view-all");
            },
            getExtraReviewerInlineDialog: function(content) {
                return {
                    rowsElement: content.find("tr"),
                    rowsValue: function() {
                        var values = [];
                        _.each(this.rowsElement, function(row) {
                            var $row = AJS.$(row);
                            values.push({
                                name: $row.find(".reviewer-name").text(),
                                hasAvatar: $row.find(".reviewer-avatar img").length > 0
                            });
                        });
                        return values;
                    }

                }
            }
        }
    },
    createView: function(count) {
        return new JIRA.DevStatus.DetailDialogPullRequestView({
            count: 10,
            issueKey: this.issueKey,
            issueId: this.issueId,
            tabs: {
                bitbucket: {name:"Bitbucket", count: count || 9}
            },
            dataType: "pullrequest",
            reviewersThreshold: 2
        });
    },
    assertNoNoPermissionMessage: function() {
        ok(this.getDetailDialog().content.find(".no-permission-to-view-all").length === 0, "There is NO no-permission-to-view message");
    },
    assertNoConnectionErrorWarning: function() {
        ok(this.getDetailDialog().content.find(".aui-message.warning:has(.connection-error)").length === 0, "There is NO connection error message");
    },
    assertNoErrorMessagesInCanvas: function() {
        this.assertNoNoPermissionMessage();
        this.assertNoConnectionErrorWarning();
    }
});

test("Test content table is not rendered when JSON response empty", function() {
    var view = this.createView();
    var jiraDialog = AJS.$('<div class="jira-dialog"></div>').appendTo(AJS.$("#qunit-fixture"));
    var container = AJS.$('<div class="devstatus-dialog-content"></div>').appendTo(jiraDialog);
    view.getContentContainer = this.sandbox.stub().returns(container);
    view.renderSuccess("bitbucket", []);

    var dialog = this.getDetailDialog();
    equal(dialog.getPullRequestContainer().length, 0, "Dialog container doesn't exists");
    equal(dialog.getRows().length, 0, "Table/rows are not rendered into the dialog");
    equal(dialog.getNoPermissionToViewAllMessage().length, 1, "Contains no permission to view all message");
});

test("Test lock screen renders if no data would be displayed", function() {
    var applicationType = 'fecru';
    var data = {detail: [{pullRequests: []}] };
    var view = this.createView();

    var $fixture = AJS.$('#qunit-fixture');
    $fixture.html('<div id="tab-content-' + applicationType + '"><div class="detail-content-container"></div></div>');
    view.$el = $fixture;
    view.renderSuccess = this.sandbox.spy();
    view._handleFetchSucceeded(applicationType, data);

    ok(view.renderSuccess.neverCalledWith(applicationType, data.detail));
    var contentContainer = view.getContentContainer(applicationType);
    var noPermissionToViewAll = contentContainer.find(".no-permission-to-view-all");
    ok(noPermissionToViewAll.find(".lock-image").length > 0, "has lock image");
    equal(noPermissionToViewAll.text(), "You don't have access to view all related pull requests. Please contact your administrator.");
});

test("Test author properly excluded from reviewers", function() {
    var view = this.createView();

    var pullRequests = [{reviewers:[{name:"amy", name:"bernice"}],author:{name:"amy"}}];
    var filteredPRs = view._removeAuthorFromReviewers(pullRequests);

    deepEqual(filteredPRs, [{reviewers:[{name:"bernice"}],author:{name:"amy"}}]);
});

test("Test empty reviewer PR does not error", function() {
    var view = this.createView();

    var pullRequests = [{reviewers:[],author:{name:"amy"}}];
    var filteredPRs = view._removeAuthorFromReviewers(pullRequests);

    deepEqual(filteredPRs, [{reviewers:[],author:{name:"amy"}}]);
});

test("Test reviewerless PR does not error", function() {
    var view = this.createView();

    var pullRequests = [{author:{name:"amy"}}];
    var filteredPRs = view._removeAuthorFromReviewers(pullRequests);

    deepEqual(filteredPRs, [{author:{name:"amy"}}]);
});

test("Test authorless PR is unchanged", function() {
    var view = this.createView();

    var pullRequests = [{reviewers:[{name:"amy", name:"bernice"}]}];
    var filteredPRs = view._removeAuthorFromReviewers(pullRequests);

    deepEqual(filteredPRs, [{reviewers:[{name:"amy",name:"bernice"}]}]);
});

test("Test responses from multiple connected products are combined and sorted", function() {
    var view = this.createView();
    view.template = this.sandbox.spy();
    view.getContentContainer = this.sandbox.stub().returns(AJS.$("<div></div>"));

    view.renderSuccess("product", [
        {
            pullRequests: [{id: "#999", name: "haha", lastUpdate: '2013-10-22T15:33:50.000+1100'},
                {id:"#1", name: "zzzzzzz", lastUpdate: '2013-09-22T15:33:50.000+1100'},
                {id: "#5", name: "bbbb", lastUpdate: '2013-08-22T15:33:50.000+1100'},
                {id:"#2", name: "ccccc", lastUpdate: '2013-11-22T15:33:50.000+1100'}]
        },
        {
            pullRequests: [{id: "#3", name: "eeee", lastUpdate: '2013-10-22T15:33:50.000+1102'},
                    {id:"#3", name: "dddd", lastUpdate: '2013-10-22T15:33:50.000+1100'}]
        }
    ]);

    deepEqual(view.template.getCall(0).args[0].pullRequests,[
        {id:"#5", name: "bbbb", lastUpdate: '2013-08-22T15:33:50.000+1100'},
        {id:"#1", name: "zzzzzzz", lastUpdate: '2013-09-22T15:33:50.000+1100'},
        {id: "#999", name: "haha", lastUpdate: '2013-10-22T15:33:50.000+1100'},
        {id:"#3", name: "dddd", lastUpdate: '2013-10-22T15:33:50.000+1100'},
        {id: "#3", name: "eeee", lastUpdate: '2013-10-22T15:33:50.000+1102'},
        {id:"#2", name: "ccccc", lastUpdate: '2013-11-22T15:33:50.000+1100'}
    ].reverse());
});

test("Test list of pull requests are ordered by pull request id and name", function() {
    var view = this.createView();
    deepEqual(view._getSortedPullRequestsByDate(
                          [{id: "#999", name: "haha", lastUpdate: '2013-10-22T15:33:50.000+1100'},
                           {id:"#1", name: "zzzzzzz", lastUpdate: '2013-09-22T15:33:50.000+1100'},
                           {id: "#5", name: "bbbb", lastUpdate: '2013-08-22T15:33:50.000+1100'},
                           {id:"#2", name: "ccccc", lastUpdate: '2013-11-22T15:33:50.000+1100'},
                           {id: "#3", name: "eeee", lastUpdate: '2013-10-22T15:33:50.000+1102'},
                           {id:"#3", name: "dddd", lastUpdate: '2013-10-22T15:33:50.000+1100'}]
    ),[
        {id:"#5", name: "bbbb", lastUpdate: '2013-08-22T15:33:50.000+1100'},
        {id:"#1", name: "zzzzzzz", lastUpdate: '2013-09-22T15:33:50.000+1100'},
        {id: "#999", name: "haha", lastUpdate: '2013-10-22T15:33:50.000+1100'},
        {id:"#3", name: "dddd", lastUpdate: '2013-10-22T15:33:50.000+1100'},
        {id: "#3", name: "eeee", lastUpdate: '2013-10-22T15:33:50.000+1102'},
        {id:"#2", name: "ccccc", lastUpdate: '2013-11-22T15:33:50.000+1100'}
    ].reverse());
});

test("Test sorting reviewers in pull requests", function() {
    var view = this.createView();
    var pullRequests = [
        {id: "#991", name: "haha", lastUpdate: '2013-10-22T15:33:50.000+1100',
            reviewers: [ {name: "A", approved: false}, {name: "B", approved: true}, {name: "C", approved: false}]
        },
        {id: "#992", name: "haha", lastUpdate: '2013-10-22T15:33:50.000+1100',
            reviewers: [ {name: "A"}, {name: "B", approved: true}, {name: "C", approved: false}]
        },
        {id: "#993", name: "haha", lastUpdate: '2013-10-22T15:33:50.000+1100',
            reviewers: [ {name: "C", approved: false}, {name: "B", approved: false}, {name: "A", approved: false}]
        },
        {id: "#995", name: "haha", lastUpdate: '2013-10-22T15:33:50.000+1100',
            reviewers: [ {name: "C", approved: true}, {name: "B", approved: true}, {name: "A", approved: true}]
        },
        {id: "#994", name: "haha", lastUpdate: '2013-10-22T15:33:50.000+1100',
            reviewers: [ {approved: false}, {approved: true}, {name: "C", approved: false}]
        }
    ];
    view._sortReviewers(pullRequests);
    deepEqual(pullRequests, [
        {id: "#991", name: "haha", lastUpdate: '2013-10-22T15:33:50.000+1100',
            reviewers: [ {name: "B", approved: true}, {name: "A", approved: false}, {name: "C", approved: false}]
        },
        {id: "#992", name: "haha", lastUpdate: '2013-10-22T15:33:50.000+1100',
            reviewers: [ {name: "B", approved: true}, {name: "A"}, {name: "C", approved: false}]
        },
        {id: "#993", name: "haha", lastUpdate: '2013-10-22T15:33:50.000+1100',
            reviewers: [ {name: "A", approved: false}, {name: "B", approved: false}, {name: "C", approved: false}]
        },
        {id: "#995", name: "haha", lastUpdate: '2013-10-22T15:33:50.000+1100',
            reviewers: [ {name: "A", approved: true}, {name: "B", approved: true}, {name: "C", approved: true}]
        },
        {id: "#994", name: "haha", lastUpdate: '2013-10-22T15:33:50.000+1100',
            reviewers: [ {name: "C", approved: false}]
        }
    ]);
});

test("Test detail pullrequest dialog with success request", function() {
    expect(3 + 2 + 3 + (13 * 6));
    JIRA.DevStatus.Date.format = 'YYYY-MM-DD';
    var view = this.createView(5);
    view.show();

    var dialog = this.getDetailDialog();
    var spinner = dialog.content.find(".status-loading");
    var activePane = dialog.content.find(".tabs-pane.active-pane");

    ok(spinner.is(":visible"), "A spinner is visible  when there is an onflight ajax request");
    ok(spinner.children().length > 0);
    ok(activePane.hasClass("loading"), "Dialog content is faded out when there is an onflight ajax request");

    AJS.TestContractManager.respondToRequest("detailCommit3LO-stash", this.server.requests[0]);
    AJS.TestContractManager.respondToRequest("detailBranchPullRequest-multiple", this.server.requests[1]);

    this.assertNoErrorMessagesInCanvas();

    ok(!spinner.is(":visible"), "A spinner is no longer visible when the onflight ajax request has completed");
    ok(!activePane.hasClass("loading"), "Dialog content is no longer faded out when ajax has completed");
    equal(dialog.getRows().length, 6, "Number of pull request rows rendered into the table is correct");

    //Check contents are non-empty, except 2 and 5
    var emptyReviewerColumnIndex = [2, 5];
    var withCommentCountIndex = {"1": 3, "0": 1};
    _.each(dialog.getRows(), function(row, rowIndex) {
        var columns = dialog.getColumns(row);

        ok(columns.author.children().length !== 0, "Author column is not empty");
        ok(columns.author.find("img").length !== 0, "Author column has avatar");
        ok(!_.isEmpty(columns.state.text()), "State column is not empty");
        ok(!_.isEmpty(columns.title.text()), "Title column is not empty");
        ok(columns.id.find("a") != null, "ID content is a link");
        ok(columns.title.find("a") != null, "Title content is a link");
        ok(columns.title.find("a").attr("href").indexOf("https://bitbucket.org/fusiontestaccount/ddd-one/pull-request/") !== -1,
            "Title content has a link");

        if (_.contains(emptyReviewerColumnIndex, rowIndex)) {
            equal(columns.reviewer.children().length, 0, "Reviewer column is empty");
            equal(columns.reviewer.find("img").length, 0, "Reviewer column does not have avatar");
        } else {
            ok(columns.reviewer.children().length !== 0, "Reviewer column is NOT empty");
            ok(columns.reviewer.find("img").length !== 0, "Reviewer column has avatar");
        }

        var commentCount = withCommentCountIndex[""+rowIndex];
        if (commentCount) {
            equal(columns.commentCount.text(), "Comments" + commentCount, "Comment count is correct");
        } else {
            equal(columns.commentCount.children().length, 0, "Reviewer column is empty");
        }

        ok(_.size(columns.source) === 0, "No Source column");

        ok(_.size(columns.destination) === 0, "No Destination column");

        ok(!_.isEmpty(columns.timestamp.text()), "Timestamp column is not empty");
    });
});

test("Test reviewer tooltip in detail dialog", function() {
    JIRA.DevStatus.Date.format = 'YYYY-MM-DD';
    var view = this.createView(5);
    view.options.reviewersThreshold = -1; //So that everything is shown in the tooltip
    view.show();

    var dialog = this.getDetailDialog();

    AJS.TestContractManager.respondToRequest("detailCommit3LO-stash", this.server.requests[0]);
    AJS.TestContractManager.respondToRequest("detailBranchPullRequest-multiple", this.server.requests[1]);

    equal(dialog.getRows().length, 6, "Number of pull request rows rendered into the table is correct");

    var rowWithRewiewers = 3;

    //Clicking on number lozenge on the reviewer column brings up the inline dialog
    ok(!_.isObject(view.reviewerInlineDialogView.activeExtraReviewersToolTip), "activeExtraReviewersToolTip is NOT created when the pull request link is NOT hovered");
    dialog.getColumns(dialog.getRows()[rowWithRewiewers]).reviewer.find(".extrareviewers-tooltip").mouseenter().mousemove();
    ok(_.isObject(view.reviewerInlineDialogView.activeExtraReviewersToolTip), "activeExtraReviewersToolTip is created when the pull request link is hovered");

    var prInlineDialog = dialog.getExtraReviewerInlineDialog(view.reviewerInlineDialogView.activeExtraReviewersToolTip.$content);
    equal(prInlineDialog.rowsElement.length, 1, "number of reviewers in tooltip is correct");
    deepEqual(prInlineDialog.rowsValue(), [
        {name: "Jimmy Kurniawan", hasAvatar: true}
    ], "tooltip content is correct");

    dialog.getColumns(dialog.getRows()[rowWithRewiewers]).reviewer.find(".extrareviewers-tooltip").mouseleave();
    ok(!view.reviewerInlineDialogView.activeExtraReviewersToolTip.visible, "activeExtraReviewersToolTip is hidden when mouse is removed");
});

test("Test clicking pull request link in pull request detail dialog fires the correct analytics", function() {
    expect(1 + (6 * 2));
    var view = this.createView(5);
    view.show();

    var dialog = this.getDetailDialog();

    AJS.TestContractManager.respondToRequest("detailCommit3LO-stash", this.server.requests[0]);
    AJS.TestContractManager.respondToRequest("detailBranchPullRequest-multiple", this.server.requests[1]);

    equal(dialog.getRows().length, 6, "Number of pull request rows rendered into the table is correct");

    // An exhaustive click test for all the pull request link, because why not :)
    _.each(dialog.getRows(), function(row, rowIndex) {
        var links = dialog.getLinks(row);
        links.pullRequest.click();
        ok(JIRA.DevStatus.PullRequestsAnalytics.fireDetailPullRequestClicked.called,
            "Analytic event is fired correctly when clicking on pull request link on row " + (rowIndex + 1));
        ok(JIRA.DevStatus.PullRequestsAnalytics.fireDetailPullRequestClicked.calledWith("bitbucket"),
            "Analytic event is fired correctly when clicking on pull request link on row " + (rowIndex + 1));

        JIRA.DevStatus.PullRequestsAnalytics.fireDetailPullRequestClicked.reset();
    });
});
