Backbone.define("JIRA.DevStatus.DetailDialogBranchView", JIRA.DevStatus.BaseDetailDialogView.extend({
    template: JIRA.Templates.DevStatus.DetailDialog.branch,
    pullRequestToolTipTemplate: JIRA.Templates.DevStatus.DetailDialog.pullRequestToolTip,
    STATE_ORDER: ["OPEN", "MERGED", "DECLINED"],

    events: {
        "click a.repository-link": "_onClickRepositoryLink",
        "click a.branch-link": "_onClickBranchLink",
        "click a.pullrequest-link": "_onClickPullRequestLink",
        "click a.create-pullrequest-link": "_onClickCreatePullRequestLink",
        "mouseenter .pullrequest-tooltip": "_onHoverPullRequestsToolTip",
        "click .pullrequest-tooltip": "_onClickPullRequestsToolTip"
    },

    initialize: function(options) {
        this.analyticIssueData = options.analyticIssueData || {};
        this.analytics = JIRA.DevStatus.BranchesAnalytics;
        JIRA.DevStatus.BaseDetailDialogView.prototype.initialize.call(this, _.extend({}, options, {
            type: "branch",
            width: 1000,
            height: 400,
            //This is a hack to force the system to actually retrieve pull request detail json, instead of branch detail json
            //Because branch detail dialog uses the same detail json as pull request dialog and Stash doesn't have a branch detail capability.
            dataType: "pullrequest"
        }));
    },

    renderSuccess: function(applicationType, result) {
        var branches = this._sortBranchesByName(this._removeWithNoRepos(this._extractAllBranches(result)));
        var pullrequests = this._extractAllPullRequests(result);
        this._populatePullRequestStateByBranch(branches, pullrequests);
        var features = this._getBranchFeatures(branches);
        var branchCount = this._getBranchCount(branches);
        var container = this.getContentContainer(applicationType);
        if (branchCount > 0) {
            container.html(this.template({
                applicationType: applicationType,
                livestampRelativizeThreshold: this.options.livestampRelativizeThreshold,
                branches: branches,
                features: features,
                branchCount: branchCount
            }));
            JIRA.DevStatus.Date.addTooltip(container);
        } else {
            container.empty();
        }
        this.renderNoPermissionToViewAllWarningAtBottom(applicationType, branchCount);
        return this;
    },

    getTitle: function(count, issue) {
        return AJS.I18n.getText("devstatus.detail.panel.title.branch", count, issue);
    },

    getOAuthMessageInFooter: function(instances) {
        return AJS.I18n.getText('devstatus.authentication.message.branch', instances.length);
    },

    getOAuthMessageInCanvas: function() {
        return AJS.I18n.getText('devstatus.authentication.authenticate.branch.title');
    },

    getConnectionMessageInCanvas: function() {
        return AJS.I18n.getText('devstatus.authentication.connection-problem.branch.title');
    },

    getNoPermissionToViewAllMessageInCanvas: function() {
        return AJS.I18n.getText('devstatus.authentication.no-permission-to-view-all.branch.title');
    },

    hasData: function(detail) {
        return this._removeWithNoRepos(this._extractAllBranches(detail)).length > 0;
    },

    _extractAllBranches: function(result) {
        return _.compact(_.flatten(_.pluck(result, 'branches')));
    },

    _getBranchFeatures: function(branches) {
        return {
            'pullRequests' : _.find(branches, function(branch) {
                return branch.createPullRequestUrl || (branch.pullRequestState && branch.pullRequestState.total);
            }) != null,
            'lastCommitId': _.find(branches, function (branch) {
                return branch.lastCommit && branch.lastCommit.displayId;
            }) != null,
            'lastCommitTimestamp': _.find(branches, function (branch) {
                return branch.lastCommit && branch.lastCommit.authorTimestamp;
            }) != null
        };
    },

    // Removes all the branches without a repository.
    _removeWithNoRepos: function(branches) {
        return _.filter(branches, function(branch) {
            return branch.repository;
        });
    },

    _extractAllPullRequests: function(result) {
        return _.flatten(_.compact(_.pluck(result, 'pullRequests')));
    },

    _sortBranchesByName: function(branches) {
        return _.sortBy(branches, function(branch) {
            // We guarantee that the repo is non-null.
            return branch.repository.name + branch.name;
        });
    },

    _onClickRepositoryLink: function() {
        this.analytics.fireDetailRepoClicked(this._getActiveApplicationType());
    },

    _onClickBranchLink: function() {
        this.analytics.fireDetailBranchClicked(this._getActiveApplicationType());
    },

    _onClickPullRequestLink: function() {
        this.analytics.fireDetailPullRequestLozengeClick(this._getActiveApplicationType());
    },

    _onClickCreatePullRequestLink: function() {
        this.analytics.fireDetailCreatePullRequestClicked(
            this._getActiveApplicationType(),
            this.analyticIssueData.isAssignee,
            this.analyticIssueData.isAssignable
        );
    },

    _onHoverPullRequestsToolTip: function(e) {
        e.preventDefault();

        var instance = this;
        var target = AJS.$(e.target);
        var id = "pullrequest-tooltip";
        this.activePullRequestsToolTip = AJS.InlineDialog(target, id,
            function(content, trigger, showPopup) {
                var $trigger = AJS.$(trigger);
                var parent = $trigger.closest("." + id);
                content.html(instance.pullRequestToolTipTemplate({
                    pullRequests: parent.data("detail")
                }));
                // For testing to peek into the content
                instance.activePullRequestsToolTip.$content = content;
                showPopup();
            }, {
                onHover: true,
                hideDelay: 200,
                showDelay: 50,
                offsetX: -200,
                cacheContent: false,
                width: 400
            }
        );
    },

    _onClickPullRequestsToolTip: function(e) {
        if (this.activePullRequestsToolTip) {
            e.stopPropagation();
        }
    },

    _getBranchCount: function (branches) {
        return branches.length;
    },

    _populatePullRequestStateByBranch: function(branches, pullrequests) {
        var instance = this;
        _.each(branches, function(branch) {
            var pullRequestsInBranch = _.filter(pullrequests, function(pullrequest) {
                return pullrequest.source.url === branch.url;
            });
            branch.pullRequestState = instance._summarisePullRequestState(pullRequestsInBranch);
        });
        return branches;
    },

    _sortPullRequestsByStatus: function(pullRequests) {
        return _.sortBy(pullRequests, function(pullRequest) {
            return (pullRequest.status === 'OPEN' ? '2' : (pullRequest.status === 'MERGED' ? "1" : "0")) +
                    pullRequest.lastUpdated;
        }).reverse();
    },

    _getFirstStatusWithCount: function(statusCounts) {
        return _.find(this.STATE_ORDER, function(state) {
            return statusCounts[state];
        });
    },

    //Mildly sucks, this logic is duplicated from the java code
    _summarisePullRequestState: function(pullRequests) {
        var statusCounts = _.groupBy(pullRequests, function(pullRequest) {
            return pullRequest.status;
        });

        return {
            status: this._getFirstStatusWithCount(statusCounts),
            data: JSON.stringify(this._sortPullRequestsByStatus(pullRequests)),
            total: _.size(pullRequests)
        }
    }
}));