AJS.test.require("com.atlassian.jira.plugins.jira-development-integration-plugin:devstatus-pullrequest-panel-resources");
AJS.test.require("com.atlassian.jira.plugins.jira-development-integration-plugin:devstatus-panel-resources");
AJS.test.require("com.atlassian.jira.plugins.jira-development-integration-plugin:devstatus-contract-test-resource");
AJS.test.require("com.atlassian.jira.plugins.jira-development-integration-plugin:devstatus-qunit-test-utils");

module("JIRA.DevStatus.DetailDialogBranchView", {
    setup: function() {
        this.sandbox = sinon.sandbox.create();
        this.issueKey = "DEV-1";
        this.issueId = 10000;
        this.server = sinon.fakeServer.create();
        JIRA.DevStatus.QunitTestUtils.spyAnalyticMethods("Branches", this.sandbox);
    },
    teardown : function () {
        this.sandbox.restore();
        this.server.restore();
        JIRA.DevStatus.QunitTestUtils.cleanAllDialogs();
        //JIRA form dialog appends overflow: hidden to the body of the page. This is to remove it.
        AJS.$("body").css("overflow", "initial");
    },
    getDetailDialog: function() {
        var dialog = AJS.$(".jira-dialog");
        return {
            heading: dialog.find(".devstatus-dialog-heading"),
            content: dialog.find(".devstatus-dialog-content"),

            getBranchContainer: function() {
                return this.content.find(".detail-branches-container");
            },
            getRows: function() {
                return this.getBranchContainer().find(".branch-row");
            },
            getColumns: function(row) {
                var $row = AJS.$(row);
                return {
                    repository: $row.find(".repository"),
                    branch: $row.find(".branch"),
                    pullRequest: $row.find(".pullrequest"),
                    action: $row.find(".action"),
                    changeset: $row.find('.changeset'),
                    lastUpdated: $row.find('.last-updated time')
                };
            },
            getLinks: function(row) {
                var columns = this.getColumns(row);
                return {
                    repository: columns.repository.find(".repository-link"),
                    branch: columns.branch.find(".branch-link"),
                    pullRequest: columns.pullRequest.find(".pullrequest-link"),
                    createPullRequest: columns.action.find(".create-pullrequest-link"),
                    changeset: columns.changeset.find('.changeset-link')
                };
            },
            getPullRequestInlineDialog: function(content) {
                return {
                    rowsElement: content.find("tr"),
                    rowsValue: function() {
                        var values = [];
                        _.each(this.rowsElement, function(row) {
                            var $row = AJS.$(row);
                            values.push({
                                name: $row.find(".pullrequest-name").text(),
                                state: $row.find(".pullrequest-state").text()
                            });
                        });
                        return values;
                    }

                }
            },
            getNoPermissionToViewAllMessage: function() {
                return this.content.find(".no-permission-to-view-all");
            }
        }
    },
    createView: function(count) {
        this.view = new JIRA.DevStatus.DetailDialogBranchView({
            count: 10,
            issueKey: this.issueKey,
            issueId: this.issueId,
            tabs: {
                bitbucket: {name:"Bitbucket", count: count || 9}
            }
        });
        return this.view;
    },
    setAnalyticsDataForView: function(isAssignee, isAssignable) {
        this.view.analyticIssueData = {
            isAssignee: isAssignee,
            isAssignable: isAssignable
        }
    },
    assertNoNoPermissionMessage: function() {
        ok(this.getDetailDialog().content.find(".no-permission-to-view-all").length === 0, "There is NO no-permission-to-view message");
    },
    assertNoConnectionErrorWarning: function() {
        ok(this.getDetailDialog().content.find(".aui-message.warning:has(.connection-error)").length === 0, "There is NO connection error message");
    },
    assertNoErrorMessagesInCanvas: function() {
        this.assertNoNoPermissionMessage();
        this.assertNoConnectionErrorWarning();
    }
});

test("Test content table is not rendered when JSON response empty", function() {
    var view = this.createView();
    var jiraDialog = AJS.$('<div class="jira-dialog"></div>').appendTo(AJS.$("#qunit-fixture"));
    var container = AJS.$('<div class="devstatus-dialog-content"></div>').appendTo(jiraDialog);
    view.getContentContainer = this.sandbox.stub().returns(container);
    view.renderSuccess("bitbucket", []);

    var dialog = this.getDetailDialog();
    equal(dialog.getBranchContainer().length, 0, "Dialog container doesn't exists");
    equal(dialog.getRows().length, 0, "Table/rows are not rendered into the dialog");
    equal(dialog.getNoPermissionToViewAllMessage().length, 1, "Contains no permission to view all message");
});

test("Test lock screen renders if no data would be displayed", function() {
    var applicationType = 'fecru';
    var data = {detail: [{branches: []}] };
    var view = this.createView();

    var $fixture = AJS.$('#qunit-fixture');
    $fixture.html('<div id="tab-content-' + applicationType + '"><div class="detail-content-container"></div></div>');
    view.$el = $fixture;
    view.renderSuccess = this.sandbox.spy();
    view._handleFetchSucceeded(applicationType, data);

    ok(view.renderSuccess.neverCalledWith(applicationType, data.detail));
    var contentContainer = view.getContentContainer(applicationType);
    var noPermissionToViewAll = contentContainer.find(".no-permission-to-view-all");
    ok(noPermissionToViewAll.find(".lock-image").length > 0, "has lock image");
    equal(noPermissionToViewAll.text(), "You don't have access to view all related branches. Please contact your administrator.");
});

test("Test responses from multiple connected products are combined and sorted", function() {
    var view = this.createView();
    view.template = this.sandbox.spy();
    view.getContentContainer = this.sandbox.stub().returns(AJS.$("<div></div>"));

    view.renderSuccess("product", [
        {
            branches: [{name: "D", pullRequests:[], repository: []},
                {name: "C", pullRequests:[], repository: []}]
        },
        {
            branches: [{name: "B", pullRequests: [], repository: []}]
        }
    ]);

    deepEqual(view.template.getCall(0).args[0].branches, [
        {name: "B", pullRequests: [], repository: [], pullRequestState:
            {data: "[]",status: undefined,total: 0}},
        {name: "C", pullRequests: [],repository: [], pullRequestState:
            {data: "[]",status: undefined,total: 0}},
        {name: "D", pullRequests: [],repository: [], pullRequestState:
            {data: "[]",status: undefined,total: 0}}
    ]);
});

test("Test list of branches are ordered by repository name and then branch name", function() {
    var view = this.createView();
    // Note: Pull requests are not sorted here, because we filter and cross-reference these.
    deepEqual(view._sortBranchesByName([
        {
            name: "D",
            repository: { name: "B" },
            pullRequests: [{id: "#4", name: "a"}, {id: "#3", name: "b"}]
        }, {
            name: "C",
            repository: { name: "B" },
            pullRequests:[]
        }, {
            name: "C",
            repository: { name: "0" },
            pullRequests: []
        }, {
            name: "D",
            repository: { name: "0" },
            pullRequests: []
        }, {
            name: "B",
            repository: { name: "0" },
            pullRequests: []
        }
    ]),[
        {name: "B", repository: { name: "0" }, pullRequests: []},
        {name: "C", repository: { name: "0" }, pullRequests: []},
        {name: "D", repository: { name: "0" }, pullRequests: []},
        {name: "C", repository: { name: "B" }, pullRequests: []},
        {name: "D", repository: { name: "B" }, pullRequests: [{id: "#4", name: "a"}, {id: "#3", name: "b"}]}
    ]);
});

test("List of pull requests are summarised for branch", function() {
    var view = this.createView();
    deepEqual(view._populatePullRequestStateByBranch([
        {name: "D", url: "http://somelocation.com"},
        {name: "C", url: "http://somelocation2.com"},
        {name: "A", url: "http://somelocation3.com"}
    ], [
        {
            status: "MERGED",
            source: { url: "http://somelocation.com" }
        }, {
            status: "DECLINED",
            source: { url: "http://somelocation.com" }
        }, {
            status: "OPEN",
            url: "http://url.com",
            source: { url: "http://somelocation2.com" }
        }, {
            status: "MERGED",
            source: { url: "http://somelocation3.com" }
        }, {
            status: "MERGED",
            source: { url: "http://somelocation3.com" }
        }, {
            status: "OPEN",
            source: { url: "http://somelocation3.com" }
        }, {
            status: "DECLINED",
            source: { url: "http://somelocation3.com" }
        }
    ]), [
        {
            name: "D",
            url: "http://somelocation.com",
            pullRequestState: {
                data: JSON.stringify([{status: "MERGED", source: { url: "http://somelocation.com" }},
                    {status: "DECLINED", source: { url: "http://somelocation.com" }}]),
                status: "MERGED",
                total: 2
            }
        }, {
            name: "C",
            url: "http://somelocation2.com",
            pullRequestState: {
                data: JSON.stringify([{status: "OPEN", url: "http://url.com",
                    source: { url: "http://somelocation2.com" }}]),
                status: "OPEN",
                total: 1
            }
        },
        {
            name: "A",
            url: "http://somelocation3.com",
            pullRequestState: {
                data: JSON.stringify([{status: "OPEN", source: {url: "http://somelocation3.com"}},
                    {status: "MERGED", source: {url: "http://somelocation3.com"}},
                    {status: "MERGED", source: {url: "http://somelocation3.com"}},
                    {status: "DECLINED", source: {url: "http://somelocation3.com"}}]),
                status: "OPEN",
                total: 4
            }
        }
    ]);
});

test("Sort pull requests by status", function() {
    var view = this.createView();
    deepEqual(view._sortPullRequestsByStatus([
        {status: "MERGED", lastUpdated: "2013-11-15T16:52:40.081+1100"},
        {status: "DECLINED", lastUpdated: "2013-11-15T16:52:40.082+1100"},
        {status: "DECLINED", lastUpdated: "2013-11-15T16:52:40.081+1100"},
        {status: "MERGED", lastUpdated: "2013-11-15T16:52:40.082+1100"},
        {status: "MERGED", lastUpdated: "2013-11-15T16:52:40.083+1100"},
        {status: "OPEN", lastUpdated: "2013-11-15T16:52:40.081+1100"}
    ]), [
        {status: "OPEN", lastUpdated: "2013-11-15T16:52:40.081+1100"},
        {status: "MERGED", lastUpdated: "2013-11-15T16:52:40.083+1100"},
        {status: "MERGED", lastUpdated: "2013-11-15T16:52:40.082+1100"},
        {status: "MERGED", lastUpdated: "2013-11-15T16:52:40.081+1100"},
        {status: "DECLINED", lastUpdated: "2013-11-15T16:52:40.082+1100"},
        {status: "DECLINED", lastUpdated: "2013-11-15T16:52:40.081+1100"}
    ]);
});

test("Branches with null repository is not rendered", function() {
    this.issueKey = "DEV-2";
    this.issueId = 10001;
    var view = this.createView(4);
    view.show();

    AJS.TestContractManager.respondToRequest("detailCommit3LO-stash", this.server.requests[0]);
    AJS.TestContractManager.respondToRequest("detailBranchPullRequest-nullRepo", this.server.requests[1]);

    var dialog = this.getDetailDialog();
    equal(dialog.getRows().length, 0, "No branch is rendered onto the dialog");
});

test("Test detail branch dialog with success request", function() {
    expect(3 + 2 + 3 + (10 * 5) + 5);
    JIRA.DevStatus.Date.format = 'YYYY-MM-DD';
    var view = this.createView(4);
    view.show();

    var dialog = this.getDetailDialog();
    var spinner = dialog.content.find(".status-loading");
    var activePane = dialog.content.find(".tabs-pane.active-pane");

    ok(spinner.is(":visible"), "A spinner is visible  when there is an onflight ajax request");
    ok(spinner.children().length > 0);
    ok(activePane.hasClass("loading"), "Dialog content is faded out when there is an onflight ajax request");

    AJS.TestContractManager.respondToRequest("detailCommit3LO-stash", this.server.requests[0]);
    AJS.TestContractManager.respondToRequest("detailBranchPullRequest-multiple", this.server.requests[1]);

    this.assertNoErrorMessagesInCanvas();

    ok(!spinner.is(":visible"), "A spinner is no longer visible when the onflight ajax request has completed");
    ok(!activePane.hasClass("loading"), "Dialog content is no longer faded out when ajax has completed");
    equal(dialog.getRows().length, 5, "Number of branch rows rendered into the table is correct");

    //Check contents are non-empty, except 1
    var emptyPullRequestColumnIndex = [0];
    _.each(dialog.getRows(), function(row, rowIndex) {
        var columns = dialog.getColumns(row);
        ok(!_.isEmpty(columns.repository.text()), "Repository column is not empty");
        ok(columns.repository.find("a") != null, "Repository content is a link");
        equal(columns.repository.find("a").attr("href"), "https://bitbucket.org/fusiontestaccount/ddd-one", "Repository content has a link");

        var branchText = columns.branch.text();
        ok(!_.isEmpty(branchText), "Branch column is not empty");
        ok(columns.branch.find("a") != null, "Branch content is a link");
        equal(columns.branch.find("a").attr("href"), "https://bitbucket.org/fusiontestaccount/ddd-one/branch/" + encodeURIComponent(branchText),
            "Branch content has a link");

        if (_.contains(emptyPullRequestColumnIndex, rowIndex)) {
            ok(_.isEmpty(columns.pullRequest.text()), "Pullrequest column is empty");
        } else {
            ok(!_.isEmpty(columns.pullRequest.text()), "Pullrequest column is not empty");
        }

        ok(!_.isEmpty(columns.action.text()), "Action column is not empty");
        ok(columns.action.find("a") != null, "Action content is a link");
        ok(columns.action.find("a").attr("href").indexOf("https://bitbucket.org/fusiontestaccount/ddd-one/pull-request/new") !== -1,
            "Action content has a link");
    });

    //Clicking on pull request column with multiple pull request brings up the inline dialog
    ok(!_.isObject(view.activePullRequestsToolTip), "PullRequestsToolTip is NOT created when the pull request link is NOT hovered");
    dialog.getColumns(dialog.getRows()[1]).pullRequest.find(".pullrequest-tooltip").mouseenter().mousemove();
    ok(_.isObject(view.activePullRequestsToolTip), "PullRequestsToolTip is created when the pull request link is hovered");

    var prInlineDialog = dialog.getPullRequestInlineDialog(view.activePullRequestsToolTip.$content);
    equal(prInlineDialog.rowsElement.length, 3, "number of statuses in tooltip is correct");
    deepEqual(prInlineDialog.rowsValue(), [
        {name: "#6 DEV-1 everything is nice and good and i did not forget to add issue key to commit message and PR title", state: "OPEN"},
        {name: "#5 DEV-1 everything is nice and good and i did not forget to add issue key to commit message and PR title", state: "MERGED"},
        {name: "#4 DEV-1 everything is nice and good and i did not forget to add issue key to commit message and PR title", state: "DECLINED"}
    ], "tooltip content is correct");

    dialog.getColumns(dialog.getRows()[1]).pullRequest.find(".pullrequest-tooltip").mouseleave();
    ok(!view.activePullRequestsToolTip.visible, "PullRequestsToolTip is hidden when mouse is removed");
});

test("Test detail branch dialog with last commit details", function() {
    // having
    var view = this.createView(3);

    // when
    view.show();
    AJS.TestContractManager.respondToRequest("detailCommit3LO-stash", this.server.requests[0]);
    AJS.TestContractManager.respondToRequest("detailBranchPullRequest-lastCommit", this.server.requests[1]);
    var dialog = this.getDetailDialog();

    // then
    equal(dialog.getRows().length, 1, "Rendered rows");
    var row = dialog.getRows()[0];
    equal(dialog.getColumns(row).lastUpdated.attr('datetime'), "1385729932000");
    equal(dialog.getLinks(row).changeset.text(), "1f79b9a");
    equal(dialog.getLinks(row).changeset.attr('href'), "http://lpater-dev:8060/foo/changelog/gitsample?cs=1f79b9a3f334ee9db41d30193a3b3ceae7da7899");
    ok(!dialog.getColumns(row).action.length);
    ok(!dialog.getColumns(row).pullRequest.length);
});

test("Test getting branch features", function() {
    // having
    var view = this.createView(3),
        branchesWithNoFeatures = [{}],
        branchesWithPullRequests = [{
            pullRequestState: {
                total: 1
            }
        }],
        branchesWithCommits = [{
            lastCommit: {
                displayId: 'faf',
                authorTimestamp: 'abc'
            }
        }],
        branchesWithCommitIdsOnly = [{
            lastCommit: {
                displayId: 'faf'
            }
        }],
        branchesWithCommitsAndPullRequests = [{
            lastCommit:  {
                displayId: 'faf',
                authorTimestamp: 'abc'
            },
            createPullRequestUrl: "http://foo"
        }];

    // when
    noFeatures = view._getBranchFeatures(branchesWithNoFeatures);
    pullRequests = view._getBranchFeatures(branchesWithPullRequests);
    commits = view._getBranchFeatures(branchesWithCommits);
    commitsWithIdsOnly = view._getBranchFeatures(branchesWithCommitIdsOnly);
    commitsAndPullRequests = view._getBranchFeatures(branchesWithCommitsAndPullRequests);

    // then
    ok(!noFeatures.lastCommitId);
    ok(!noFeatures.lastCommitTimestamp);
    ok(!noFeatures.pullRequests);

    ok(!pullRequests.lastCommitId);
    ok(!pullRequests.lastCommitTimestamp);
    ok(pullRequests.pullRequests);

    ok(commits.lastCommitId);
    ok(commits.lastCommitTimestamp);
    ok(!commits.pullRequests);

    ok(commitsWithIdsOnly.lastCommitId);
    ok(!commitsWithIdsOnly.lastCommitTimestamp);
    ok(!commitsWithIdsOnly.pullRequests);

    ok(commitsAndPullRequests.lastCommitId);
    ok(commitsAndPullRequests.lastCommitTimestamp);
    ok(commitsAndPullRequests.pullRequests);
});

test("Test analytics for branch detail dialog", function() {
    expect(1 + (6 * 2 * 5));
    var view = this.createView(4);
    view.show();

    var dialog = this.getDetailDialog();

    AJS.TestContractManager.respondToRequest("detailCommit3LO-stash", this.server.requests[0]);
    AJS.TestContractManager.respondToRequest("detailBranchPullRequest-multiple", this.server.requests[1]);

    equal(dialog.getRows().length, 5, "Number of branch rows rendered into the table is correct");

    function assertCalls(isCalled, isCalledWith, msg) {
        ok(isCalled, msg);
        ok(isCalledWith, msg);
    }
    function checkClickLinkAnalytics (analytic, rowIndex) {
        var call = JIRA.DevStatus.BranchesAnalytics[analytic].getCall(rowIndex);
        var isCalled = _.isObject(call);
        var isCalledWith = isCalled && call.calledWith("bitbucket");
        var msg = "Analytic event is fired for " + analytic + " link on row " + (rowIndex + 1);
        assertCalls(isCalled, isCalledWith, msg);
    }
    function checkPullRequestClickAnalytics (rowIndex, negation) {
        var analytic = JIRA.DevStatus.BranchesAnalytics.fireDetailPullRequestLozengeClick;
        var isCalled = analytic.called;
        var isCalledWith = isCalled && analytic.calledWith("bitbucket");
        var extraMsg = "";
        if (negation) {
            isCalled = !isCalled;
            isCalledWith = !isCalledWith;
            extraMsg = " NOT";
        }
        var msg = "Analytics for pull request click is" + extraMsg + " called for row " + (rowIndex + 1);
        assertCalls(isCalled, isCalledWith, msg);
        analytic.reset();
    }
    function checkCreatePullRequestAnalytics (rowIndex, isAssignee, isAssignable) {
        var analytic = JIRA.DevStatus.BranchesAnalytics.fireDetailCreatePullRequestClicked;
        var msg = "Analytics for create pull request is called for row " + (rowIndex + 1) + " (" + isAssignee + ", " + isAssignable + ")";
        assertCalls(analytic.called, analytic.calledWith("bitbucket", isAssignee, isAssignable), msg);
        analytic.reset();
    }

    var instance = this;
    var emptyPullRequestColumnIndex = [0];
    _.each(dialog.getRows(), function(row, rowIndex) {
        var links = dialog.getLinks(row);

        links.repository.click();
        checkClickLinkAnalytics("fireDetailRepoClicked", rowIndex);

        links.branch.click();
        checkClickLinkAnalytics("fireDetailBranchClicked", rowIndex);

        links.pullRequest.click();
        checkPullRequestClickAnalytics(rowIndex, _.contains(emptyPullRequestColumnIndex, rowIndex));

        instance.setAnalyticsDataForView(true);
        links.createPullRequest.click();
        checkCreatePullRequestAnalytics(rowIndex, true);

        instance.setAnalyticsDataForView(false, true);
        links.createPullRequest.click();
        checkCreatePullRequestAnalytics(rowIndex, false, true);

        instance.setAnalyticsDataForView(false, false);
        links.createPullRequest.click();
        checkCreatePullRequestAnalytics(rowIndex, false, false);
    });
});
