/**
 * This is the generic abstract backbone view for the detail dialog
 *
 *
 */
Backbone.define("JIRA.DevStatus.BaseDetailDialogView", Backbone.View.extend({
    frameEvents: {
        "click .menu-item": "_doTabClick"
    },

    frameTemplate: JIRA.Templates.DevStatus.DetailDialog.frame,
    oauthStatusTemplate: JIRA.Templates.DevStatus.DetailDialog.oauthStatus,
    authenticationTemplate: JIRA.Templates.DevStatus.DetailDialog.authenticationScreen,
    connectionProblemTemplate: JIRA.Templates.DevStatus.DetailDialog.connectionProblemScreen,
    connectionProblemWarningTemplate: JIRA.Templates.DevStatus.connectionProblemAsWarning,
    noPermissionToViewAllTemplate: JIRA.Templates.DevStatus.DetailDialog.noPermissionToViewAll,

    /**
     * @param options
     * @param options.id {String} Dialog id
     * @param options.issueKey {String} Issue key
     * @param options.issueId {String} Issue id
     * @param options.count {Number} Number of the associated type
     * @param options.model {Backbone.Model} Optional, if none is given, then JIRA.DevStatus.BaseDetailDialogModel will be used
     * @param options.tabs {[{name: String, count: Number}]} A list of map containing name, type and count
     * @param options.type {String} The type of this dialog. Will be used as part of i18n.
     * @param options.dataType {String} the type of remote data (different from type)
     * @param options.width {Number} Width of the content area (does not include frame)
     * @param options.height {Number} Height of the content area (does not include frame)
     * @param options.initialTab {String} The application type of the tab to be active on open
     */
    initialize: function (options) {
        this.options = _.defaults({}, options, {
            width: 560,
            model: new JIRA.DevStatus.BaseDetailDialogModel(options)
        });
        this.model = this.options.model;
        this._initFetchHandlers();

        this._initFormDialog();
    },

    // ===============================================================================
    // The following methods are expected to be extended by the children of this class
    // ===============================================================================
    /**
     * Render the data into the canvas. Note that this is not necessarily the active pane.
     * It is possible that the user switches tab before the data for the tab comes back, in which case
     *  we are rendering the pane of a tab that's not active.
     */
    renderSuccess: function(applicationType, result) {
        return this;
    },
    /**
     * By default, visiting a tab in the detail dialog will cause JIRA.DevStatus.BaseDetailDialogModel to cache the content
     * of the detail dialog. This is to indicate whether it should make another ajax call.
     */
    shouldRefreshOnTabSwitch: function(applicationType) {
        return false;
    },
    /**
     * Returns the message to be displayed in the title of the detailed dialog.
     * @param count The combined count across all the instances of this entity
     * @param issue The key of the issue currently being displayed
     * @returns {String}
     */
    getTitle: function(count, issue) {
        // This should never happen
        console.warn('Unimplemented getTitle() function', this);
        return issue;
    },
    /**
     * Returns the message to be displayed in the footer to inform users that some instances are not authenticated.
     * Override this to display a message specific to the type of dev status, e.g.,
     * "You might be able to see more commits by authenticating with the following servers:"
     * @param instances [Object] the instances that are not authenticated yet. Could be useful for determining plurality.
     * @returns {String}
     */
    getOAuthMessageInFooter: function(instances) {
        return AJS.I18n.getText('devstatus.authentication.message.generic', instances.length);
    },
    /**
     * Returns the message to be displayed in the main canvas of the detailed dialog, when all instances of the current type
     * are not authenticated.
     * Override this to display a message specific to the type of dev status, e.g.,
     * "Authenticate to see more commits"
     * @returns {String}
     */
    getOAuthMessageInCanvas: function() {
        return AJS.I18n.getText('devstatus.authentication.authenticate.generic.title');
    },
    /**
     * Returns the message to be displayed in the main canvas of the detailed dialog, when all instances of the current type
     * are not authenticated.
     * Override this to display a message specific to the type of dev status, e.g.,
     * "Unable to retrieve commit information"
     * @returns {String}
     */
    getConnectionMessageInCanvas: function() {
        return AJS.I18n.getText('devstatus.authentication.connection-problem.generic.title');
    },
    /**
     * Returns the message to be displayed in the main canvas of the detailed dialog, when the user does not have permission
     * to view all dev status even after being authenticated.
     * Override this to display a message specific to the type of dev status, e.g.,
     * "You don't have access to view all related commits. Please contact your administrator."
     * @returns {String}
     */
    getNoPermissionToViewAllMessageInCanvas: function() {
        return AJS.I18n.getText('devstatus.authentication.connection-problem.generic.title');
    },
    /**
     * Returns the message to be displayed when there are permission / connection problems.
     *
     * Either a contact administrators message for logged in users, or a log in message otherwise.
     *
     * @returns {String}
     */
    getContactAdmistratorsOrLoginMessageInCanvas: function() {
        if (JIRA.Users.LoggedInUser.isAnonymous()) {
            return AJS.I18n.getText('devstatus.authentication.no-permission-to-view-all.generic.login');
        } else {
            return AJS.I18n.getText('devstatus.authentication.no-permission-to-view-all.generic.admin-contact');
        }
    },

    /**
     * Checks whether the detailed data has some data for the given detail dialog.
     * @param detail {Array} The detail json object.
     * @returns {boolean} true if there is some data for the given detail dialog.
     */
    hasData: function(detail) {
        return detail.length > 0;
    },

    /* ============================================================================= */

    show: function() {
        this.dialog.show();
    },

    hide: function() {
        this.dialog.hide();
    },

    _getPane: function(applicationType) {
        // http://learn.jquery.com/using-jquery-core/faq/how-do-i-select-an-element-by-an-id-that-has-characters-used-in-css-notation/
        return applicationType ? this.$el.find( "#tab-content-" + applicationType.replace(/(:|\.|\[|\])/g, "\\$1")) : AJS.$();
    },

    /**
     * Get the content container for the given application type (tab).
     * If the applicaitonType is not specified, get the current one.
     */
    getContentContainer: function(applicationType) {
        return this._getPane(applicationType || this._getActiveApplicationType()).find(".detail-content-container");
    },

    _doTabClick: function(e) {
        e.preventDefault();
        var appType = this._getActiveApplicationType();
        this.model.switchTab(appType, this.shouldRefreshOnTabSwitch(appType));
        this.analytics && this.analytics.fireDetailTabClicked(appType);
    },

    _redirectToLoginIf401: function (xhr) {
        if (xhr && xhr.status === 401) {
            // not logged in or session expired
            // redirect to login page by reloading the page
            JIRA.DevStatus.Navigate.redirectToLogin();
            return true;
        }
        return false;
    },

    _initFetchHandlers: function () {
        this.model.on("fetchRequested", this._showLoadingIndicator, this);
        this.model.on("fetchSucceeded", this._handleFetchSucceeded, this);
        this.model.on("fetchFailed", this._handleFetchFailed, this);
    },

    _handleFetchSucceeded: function(applicationType, result) {
        this._hideLoadingIndicator(applicationType);
        // If there are any details show them, otherwise there was an error
        var hasData = result.detail && this.hasData(result.detail);
        if (hasData) {
            this.renderSuccess(applicationType, result.detail);
        }
        // always try to render error regardless of whether there are data displayed
        this._renderError(applicationType, result.errors, hasData);
        this._postRender();
    },

    _handleFetchFailed: function(applicationType, xhr) {
        this._hideLoadingIndicator(applicationType);
        if (!this._redirectToLoginIf401(xhr)) {
            this._renderError(applicationType, [], false, true); // indicate ajaxFailed to render as generic connection problem
        }
    },

    _initAuthenticationCallbacks: function () {
        this._patchAJSIconsForAppLinks();
        // setup handlers for oauth callback (via applinks)
        var instance = this;
        var oauthApprovedHandler = function(e, applinkProperties) {
            // notify the model about this change so that it could initiate the refresh if needed
            instance.model.oauthStatusChanged(applinkProperties.appType);
            // refresh authentication status by initiating the api call again
            instance.model.fetchAuthenticationStatuses();
            // stop the default applinks handler to reload the window
            e.preventDefault();
        };
        var oauthFailedHandler = function(e, applinkProperties) {
            // there could be different reasons why the denied event is fired
            // usually it's about some errors or the user explicitly denied
            // We only handle one particular case here:
            //  When the user's session expired, we redirect to the login page
            AJS.$.ajax(AJS.contextPath() + "/rest/auth/1/session").promise()
                    .fail(function(xhr) {
                        instance._redirectToLoginIf401(xhr);
                    });
        };
        var $document = AJS.$(document);
        $document.bind('applinks.auth.approved', oauthApprovedHandler);
        $document.bind('applinks.auth.denied', oauthFailedHandler);

        // clean up the oauth callback upon dialog close
        AJS.$(this.dialog).bind("Dialog.beforeHide", function() {
            $document.unbind('applinks.auth.approved', oauthApprovedHandler);
            $document.unbind('applinks.auth.denied', oauthFailedHandler);

            // clean up the inline dialog if it exists
            instance.instanceListPopup && instance.instanceListPopup.remove();
        });

        // create a hidden banner div.applinks-auth-confirmation-container so that the confirmation msg is not shown
        AJS.$('<div class="applinks-auth-confirmation-container hidden"></div>').appendTo(this.$el);
    },

    _initFetchAuthenticationHandlers: function () {
        this.model.on("fetchAuthRequested", this._showLoadingIndicator, this);
        this.model.on("fetchAuthSucceeded", this._handleAuthSucceeded, this);
        this.model.on("fetchAuthFailed", this._handleAuthFailed, this);
    },

    _handleAuthSucceeded: function (result) {
        var instance = this;
        // get only those that are not authenticated
        var unauthInstances = _.filter(result, function(instanceWithStatus) {
            // only care about those that are not authenticated, and have an application link id
            return instanceWithStatus.configured && 
                   !instanceWithStatus.authenticated &&
                   instanceWithStatus.source &&
                   instanceWithStatus.source.applicationLinkId;
        });
        var statusDiv = this._getOAuthStatusDiv();
        if (unauthInstances.length > 0 && !JIRA.Users.LoggedInUser.isAnonymous()) {
            statusDiv.html(this.oauthStatusTemplate({
                message: this.getOAuthMessageInFooter(unauthInstances)
            }));
            this._renderUnauthInstancesInFooter(unauthInstances);
            statusDiv.show();
        } else {
            statusDiv.hide();
        }
    },

    _handleAuthFailed: function (xhr) {
        this._redirectToLoginIf401(xhr);
        // silently ignore this error
        this._getOAuthStatusDiv().hide();
    },

    _renderUnauthInstancesInFooter: function(unauthInstances) {
        var statusDiv = this._getOAuthStatusDiv();
        var appLinkDivs = _.map(_.pluck(unauthInstances, 'source'), this._createOAuthInstanceDiv);
        if (appLinkDivs.length > 0) {
            statusDiv.append(appLinkDivs[0]);
            if (appLinkDivs.length > 1) {
                statusDiv.append(', ');
                statusDiv.append(appLinkDivs[1]);
                if (appLinkDivs.length > 2) {
                    statusDiv.append(' ');
                    // show more link
                    var moreLink = AJS.$('<span class="more-instances"><a href="#">' +
                            AJS.I18n.getText('devstatus.authentication.message.more.link')+ '</a></span>');
                    statusDiv.append(moreLink);
                    var inlineDialogContent = AJS.$('<ul class="instance-list"></ul>');
                    _.each(appLinkDivs.slice(2), function(appLinkDiv) {
                        inlineDialogContent.append(AJS.$('<li class="instance-in-popup"></li>').append(appLinkDiv))
                    });
                    AJS.InlineDialog(moreLink, 'instance-list-popup',
                        function(content, trigger, showPopup) {
                            content.html(inlineDialogContent);
                            showPopup();
                            return false;
                        }, {
                            width: 150,
                            offsetX: -100,
                            onTop:true
                        }
                    );
                }
            }
        }
    },

    _createOAuthInstanceDiv: function(instance) {
        var div = ApplinksUtils.createAuthRequestInline(null, {
            id: instance.applicationLinkId,
            authUri: AJS.contextPath() + '/plugins/servlet/applinks/oauth/login-dance/authorize?applicationLinkID='
                    + encodeURIComponent(instance.applicationLinkId),
            appUri: instance.baseUrl,
            appName: instance.name,
            appType: instance.type
        });
        return AJS.$('<span class="instance"></span>').append(div.find('a.applink-authenticate').text(instance.name));
    },

    /**
     * Not needed once APL-1106 is done and included in jira core
     * @private
     */
    _patchAJSIconsForAppLinks: function() {
        AJS.icons = AJS.icons || {};
        AJS.icons.addIcon = AJS.icons.addIcon || {};
        AJS.icons.addIcon.init = AJS.icons.addIcon.init || function() {};
    },

    _getOAuthStatusDiv: function() {
        var oauthStatusDiv = this.$el.find(".buttons-container .oauth-status");
        if (oauthStatusDiv.length) {
            return oauthStatusDiv;
        } else {
            // create it if it's not there
            // insert it after the buttons, since it's not easy to make it left aligned if inserted before the buttons
            return AJS.$('<div class="oauth-status" />').appendTo(this.$el.find(".buttons-container.form-footer"));
        }
    },

    _renderError: function(applicationType, errors, hasData, ajaxFailed) {
        /**
         * split the errors into groups. Currently this includes: 'unauthorized', 'incapable' and others
         */
        var groups = _.groupBy(errors, function(errorInstance) {
            var errorType = errorInstance.error;
            if (errorType === 'unauthorized' || errorType === 'incapable') {
                return errorType
            } else {
                return 'others';
            }
        });

        /**
         * Plucking the relevant error type from the groups to show in the UI
         * Currently we only show errors for unauthorized and others.
         * Incapable errors are not shown in the detail dialog.
         */
        var unauth = _.pluck(groups['unauthorized'], 'instance');
        var others = _.pluck(groups['others'], 'instance');

        if (unauth.length > 0) {
            // trigger a refresh of the authentication message in the footer
            this.model.fetchAuthenticationStatuses();
        }
        if (hasData) {
            if (others.length > 0) {
                // prepend warning if there's any connection error
                this._renderConnectionErrorBeforeData(applicationType, others);
            }
        } else {
            // we would take over the whole screen to display error, unless hasData
            if (unauth.length > 0) {
                this._renderAuthenticationError(applicationType, unauth, others);
            } else {
                if (ajaxFailed || others.length > 0) {
                    // render the connection problem screen only when hasData is false
                    this._renderConnectionError(applicationType, others);
                } else {
                    // if not ajaxFailed and not errors at all, there must be some data that the user is not allowed to see
                    this._renderNoPermissionToViewAllWarning(applicationType);
                }
            }
        }
    },

    _renderConnectionErrorBeforeData: function(applicationType, otherInstances) {
        this.getContentContainer(applicationType).prepend(this.connectionProblemWarningTemplate({
            instances: otherInstances,
            showContactAdminForm: this.options.showContactAdminForm
        }));
    },

    _renderConnectionError: function(applicationType, otherInstances) {
        this.getContentContainer(applicationType).html(this.connectionProblemTemplate({
            instances: otherInstances,
            title: this.getConnectionMessageInCanvas(),
            showContactAdminForm: this.options.showContactAdminForm
        }));
    },

    /**
     * Render the no permission to view all warning if necessary.
     * @param applicationType the current application type
     * @param totalItemRendered the total number of data rendered on the canvas, e.g., total number of commits.
     */
    renderNoPermissionToViewAllWarningAtBottom: function(applicationType, totalItemRendered) {
        var tab = this.model.get("tabs")[applicationType] || {};
        var content = this.model.get("contentMap")[applicationType] || {};

        if (totalItemRendered < tab.count) {
            if ((content.errors || []).length === 0 || JIRA.Users.LoggedInUser.isAnonymous())  {
                // if no errors and total number of data is smaller than the count from summary
                //  then we show the warning that the user might not have permission to see all data
                // OR
                // if the user is anonymous, we will show a login message
                this.getContentContainer(applicationType).append(this._getNoPermissionToViewAllHtml());
            }
        }
    },

    _getNoPermissionToViewAllHtml: function () {
        return this.noPermissionToViewAllTemplate({
            message: this.getNoPermissionToViewAllMessageInCanvas(),
            secondaryMessage: this.getContactAdmistratorsOrLoginMessageInCanvas()
        });
    },

    _renderNoPermissionToViewAllWarning: function(applicationType) {
        this.getContentContainer(applicationType).html(this._getNoPermissionToViewAllHtml());
    },

    _renderAuthenticationError: function(applicationType, unauthInstances, otherInstances) {
        var container = this.getContentContainer(applicationType);
        container.html(this.authenticationTemplate({
            unauthInstances: unauthInstances,
            title: this.getOAuthMessageInCanvas(),
            otherInstances: otherInstances,
            showContactAdminForm: this.options.showContactAdminForm
        }));
        var appLinkDivs = _.map(unauthInstances, this._createOAuthInstanceDiv);
        var instancesDiv = container.find(".instances");
        _.each(appLinkDivs, function(appLinkDiv, index) {
            if (index !== 0) {
                // add comma as separator if not the first one
                instancesDiv.append(", ");
            }
            instancesDiv.append(appLinkDiv);
        });
    },

    _initFormDialog: function () {
        var instance = this;
        this.dialog = new JIRA.FormDialog ({
            id: this.options.id,
            width: this.options.width,
            content: function (ready) {
                instance.setElement(this.$popup);
                instance.delegateEvents(_.extend({}, instance.frameEvents, instance.events));

                instance._renderFrame();

                instance._initAuthenticationCallbacks();
                instance._initFetchAuthenticationHandlers();
                instance.model.fetchAuthenticationStatuses();
                instance.model.switchTab(instance._getActiveApplicationType());

                ready();
            },
            autoClose: true
        });
    },

    _postRender: function () {
        this.$el.find(".extra-content-in-title").tooltip();
        this.dialog._positionInCenter();
        // Select the first link element in the active tab
        this.$el.find(".active-pane a:eq(0)").focus();
    },

    _getActiveApplicationType: function() {
        return this.$el.find(".menu-item.active-tab").data("applicationType");
    },

    _showLoadingIndicator: function (applicationType) {
        var targetPane = this._getPane(applicationType);
        targetPane._removeClass("ready");
        targetPane.addClass("loading");
        var loadingIndicator = targetPane.find(".status-loading");
        loadingIndicator.spin('large');
        loadingIndicator.show();
    },

    _hideLoadingIndicator: function (applicationType) {
        var targetPane = this._getPane(applicationType);
        targetPane.addClass("ready");
        var loadingIndicator = targetPane.find(".status-loading");
        if (loadingIndicator) {
            targetPane.removeClass("loading");
            loadingIndicator.hide();
            loadingIndicator.spinStop();
        }
    },

    _renderFrame: function() {
        this.$el.html(this.frameTemplate({
            title: this.getTitle(this.options.count, this.options.issueKey),
            tabs: this._convertTabsForSoy(),
            initialTab: this.options.initialTab
        }));

        if (this.options.height) {
            this.$el.find(".form-body").css("min-height", this.options.height);
        }
    },

    _convertTabsForSoy: function() {
        /*
         Tabs given by model is in the form of:
         {
            "application": {
                count: number
            }
         }
         This is to convert the above map into an array so foreach in soy works
         */
        return _.chain(this.model.get("tabs"))
            .map(function (data, name) {
                return _.extend({
                    type: name
                }, data);
            })
            .sortBy(function(data) {
                return data.type;
            })
            .value();
    },

    /**
     * Create an instance of JIRA.DevStatus.ReviewerInlineDialogView
     */
    _createReviewersInlineDialog: function() {
        this.reviewerInlineDialogView = new JIRA.DevStatus.ReviewerInlineDialogView({
            el: this.$el.find(".extrareviewers-tooltip"),
        });
        return this.reviewerInlineDialogView;
    },

    _removeInlineDialog: function(inlineDialog) {
        if (this[inlineDialog]) {
            this[inlineDialog].hide();
            this[inlineDialog].remove();
            this[inlineDialog] = undefined;
        }
    }
}));
