if (!JIRA.DevStatus) {
    AJS.namespace("JIRA.DevStatus");
}

JIRA.DevStatus.QunitTestUtils = {
    /**
     * Spy all analytics method defined by {@code analytic}
     *
     * @param analytic type of analytic events
     * @param sandbox sinon.sandbox object
     */
    spyAnalyticMethods: function (analytic, sandbox) {
        var analyticName = analytic + "Analytics";
        if (_.isObject(JIRA.DevStatus[analyticName])) {
            this.fakeMethods(JIRA.DevStatus[analyticName], "spy", sandbox);
        } else {
            console.error("Analytics for " + analytic + " does not exist. Unable to find JIRA.DevStatus." + analyticName);
        }
    },

    /**
     * sandbox.spy() or sandbox.stub() all methods in {@code methodsMap}
     *
     * @param methodsMap map of all methods to spy or stub
     * @param fakeMethod which fake method to apply. Either spy or stub
     * @param sandbox sinon.sandbox object
     * @returns {*}
     */
    fakeMethods: function (methodsMap, fakeMethod, sandbox) {
        if (fakeMethod !== "spy" && fakeMethod !== "stub") {
            return null;
        }
        _.each(methodsMap, function(method, name) {
            methodsMap[name] = sandbox[fakeMethod]();
        });
        return methodsMap;
    },

    /**
     * Stub out defer from underscore to run the method immediately for testing purpose.
     */
    mockDefer: function() {
        var originalDefer = _.defer;
        _.defer = function(func) {
            func();
        };
        return {
            restore: function() {
                _.defer = originalDefer;
            }
        }
    },

    /**
     * Wrap the test with a fake timer and assign it to this.clock.
     * Might not be needed when sinon-qunit is integrated into atlassian-qunit: https://jdog.jira-dev.com/browse/INC-55
     * QUnit would setup the <code>this</code> reference properly.
     * Usage:
     * <code>
     *     test("test case", withFakeTimer(function(clock) {
     *         var value = 1;
     *         setTimeout(function() {value = 2;}, 1);
     *         clock.tick(1);
     *         equal(value, 2, "value should be set to 2");
     *     }));
     * </code>
     * @param testMethod the test method to be executed. Expected to accept the fake clock as the only parameter.
     * @return a no-param function to be executed by QUnit
     */
    withFakeTimer: function(testMethod) {
        return function() {
            var clock = sinon.useFakeTimers();
            try {
                testMethod.call(this, clock);
            } finally {
                clock.restore();
            }
        };
    },

    /**
     * Removes all modal (AUI or JIRA) and inline dialogs inserted into the page by QUnit tests.
     */
    cleanAllDialogs: function() {
        if (AJS.InlineDialog.current) {
            AJS.log("hiding AJS.InlineDialog.current:");
            AJS.log(AJS.InlineDialog.current);
            var clock = sinon.useFakeTimers();
            try {
                AJS.InlineDialog.current.hide();
                clock.tick(1); // to force synchronous hiding of the inline dialog
            } finally {
                clock.restore();
            }
        }
        if (AJS.Dialog.current) {
            AJS.Dialog.current.hide();
        }
        if (JIRA.Dialog.current) {
            AJS.log("hiding JIRA.Dialog.current:");
            AJS.log(JIRA.Dialog.current);
            JIRA.Dialog.current.hide();
        }
        _.each([".jira-dialog", ".aui-blanket", ".jira-page-loading-indicator", ".aui-inline-dialog"], function(selector) {
            AJS.$(selector).remove();
        });
    }
};