define('util/tracking', ['underscore', 'zepto', 'util/app-data'], function (_, $, appData) {

    /*** Config ***/

    // This are overridden if the appData key "analytics-enabled" is set to false
    var USE_ATLASSIAN_ANALYTICS = true;

    /*** Common Interface ***/

    var analyticsEnabled = appData.getBoolean('analytics-enabled');

    function elementToString(elem) {
        var str = elem.nodeName.toLowerCase();
        if (elem.id) {
            str += '#' + elem.id;
        } else {
            if (elem.classList.length) {
                str += '.' + _.toArray(elem.classList).join('.');
            }
        }
        return str;
    }

    var mappers = [];
    var tracker = {
        trackPageview: function (page) {
            _.each(mappers, function (mapper) {
                mapper.pageview(page);
            });
        },

        trackEvent: function (noun, verb, data) {
            _.each(mappers, function (mapper) {
                mapper.event(noun, verb, data);
            });
        },

        trackClick: function (element) {
            var selector = elementToString(element);
            this.trackEvent('element', 'click', {selector: selector});
        }
    };

    // Track all generic "clicks"
    $(function () {
        // Using old-school DOM method to bind on event capture phase instead of bubbling
        document.body.addEventListener('click', function (e) {
            var $target = $(e.target);
            // If element is a descendant of a link or button, track the link/button instead
            var $parent = $target.closest('a, button');
            if ($parent.length) {
                $target = $parent;
            }

            tracker.trackClick($target[0]);
        }, true);
    });

    // Track route changes
    Backbone.history.on('route', function () {
        tracker.trackPageview(Backbone.history.fragment);
    });

    /*** Atlassian Analytics ***/

    if (USE_ATLASSIAN_ANALYTICS && analyticsEnabled) {
        AJS.EventQueue = AJS.EventQueue || [];

        mappers.push({
            pageview: function (page) {
                this.event('page', 'view', {page: page});
            },
            event: function (noun, verb, data) {
                var name = ['mobile', noun, verb].join('.');
                var obj = {name: name};
                if (data) {
                    obj.properties = data;
                }
                // Stolen from jira-issue-nav-plugin/src/main/resources/content/js/util/ClientAnalytics.js
                var logMsg = "***** Analytics log [" + name + "]";
                if (data) {
                    logMsg += "[" + JSON.stringify(data) + "]";
                }
                AJS.log(logMsg);
                // Do the actual tracking
                AJS.EventQueue.push(obj);
            }
        });
    }

    return tracker;

});