define('util/ajax',
    ['zepto', 'underscore', 'backbone', 'util/app-data', 'util/router', 'exports'],
    function ($, _, Backbone, appData, router, exports) {

        /*** CUSTOM BACKBONE METHODS ***/

        var currentXhrs = [];

        function xhrDone(xhr) {
            // Quickly find an XHR promise
            var index = _.indexOf(currentXhrs, xhr);
            // Fallback in case a raw XHR was provided
            if (index === -1) {
                _.each(currentXhrs, function (xhrPromise, i) {
                    if (xhrPromise.origXHR === xhr) {
                        index = i;
                        return false;
                    }
                });
            }
            if (index >= 0) {
                currentXhrs.splice(index, 1);
            }
            // Hide loading indicator if there are no more requests
            if (!currentXhrs.length) {
                router.presenter.hideLoading();
            }
        }

        Backbone.ajax = function (settings) {
            return ajax(settings);
        };

        /**
         * Override of Backbone's sync method to provide several new features:
         *
         * - Show full page loading indicator as soon as the server is contacted
         * - Keep track of AJAX requests in flight so they can be cancelled
         *
         * Delegates to the original Backbone.sync for the core URL/method logic
         */
        Backbone.sync = _.wrap(Backbone.sync, function (_sync, method, model, options) {
            // Show the loading indicator
            if (options.showLoading !== false) {
                router.presenter.showLoading();
            }

            if (model instanceof Backbone.Collection) {
                /**
                 * If a collection.fetch() XHR is aborted after data has started to be returned by the server,
                 *   Zepto returns a 200 OK but with null data.
                 * This causes the collection to be incorrectly reset with no models.
                 *
                 * Our solution is to catch that use case and ignore any further processing, since the request
                 *   was aborted because we DON'T want to show that data.
                 */
                options.success = _.wrap(options.success, function (_success, collection, resp, settings) {
                    if (resp === null) {
                        return;
                    }
                    var args = _.rest(arguments);
                    return _success.apply(this, args);
                });
            }

            return _sync.call(this, method, model, options);
        });

        /**
         * Cache the first error handler passed to a Collection or Model (e.g. from a route).
         * This allows the same error handler to be used when calling fetch() from a refresh.
         */
        _.each(['Collection', 'Model'], function (name) {
            var proto = Backbone[name].prototype;
            proto.fetch = _.wrap(proto.fetch, function (_fetch, options) {
                options = options ? _.clone(options) : {};
                if (options.error) {
                    this._errorHandler = options.error;
                } else if (this._errorHandler) {
                    options.error = this._errorHandler;
                }

                // Check if the model/collection wants to add to the request data
                var modelUrlData = _.result(this, 'urlData');
                if (modelUrlData) {
                    options.data = _.extend(options.data || {}, modelUrlData);
                }

                // Check if the model/collection wants to add to the AJAX options
                var modelOptions = _.result(this, 'ajaxOptions');
                if (modelOptions) {
                    _.extend(options, modelOptions);
                }

                return _fetch.call(this, options);
            });
        });


        /*** PUBLIC EXPORTS ***/

        /**
         * Cancel all active XHRs - initiated by a user pressing "Cancel" on a loading spinner.
         * This is so that super-slow 3G networks (*cough Vodafone cough*) don't lock the application indefinitely.
         */
        function cancelActiveRequests() {
            _.each(currentXhrs, function (xhrPromise) {
                xhrPromise.abort();
            });
            currentXhrs = [];
            router.presenter.hideLoading();
            if (Backbone.history.fragment != router.presenter.currentFragment) {
                router.goBackSilent();
            }
        }

        /**
         * Base method for any AJAX call used in JIRA Mobile - adds several standardised AJAX options:
         *
         * - Prepend content path to URLs that start with /
         * - JSON parse responseText for error callbacks
         * - XHR tracking for user-initiated cancelling
         * - Handle aborted XHRs properly
         *
         * @param options
         * @return {xhrPromise}
         */
        function ajax(options) {
            // Context path-aware URL handling
            if (options.url && options.url.substr(0, 1) === '/') {
                var newUrl = appData.get('context-path');
                if (options.url.substr(0, 5) !== '/rest') {
                    newUrl += '/rest/mobile/1.0';
                }
                options.url = newUrl + options.url;
            }

            // Delay success callback to allow the onabort() callback to fire, if applicable
            var _success = options.success;
            if (_success) {
                options.success = function () {
                    var context = this, args = arguments;
                    _.defer(function () {
                        _success.apply(context, args);
                    });
                };
            }

            // JSON parse any error responses
            var _error = options.error;
            if (_error) {
                options.error = function (xhr) {
                    var data = xhr.responseText;
                    if (data) {
                        try {
                            data = JSON.parse(data);
                        } catch (e) {}
                    }
                    var context = this;
                    var args = _.toArray(arguments);
                    args.push(data);
                    // Allow time for the onabort() callback to fire
                    _.defer(function () {
                        _error.apply(context, args);
                        handleAjaxError(xhr, options, data);
                    });
                }
            }

            // Clean up XHR tracking on complete
            var _complete = options.complete;
            options.complete = function (xhr) {
                xhrDone(xhr);
                var context = this, args = arguments;
                _complete && _.defer(function () {
                    _complete.apply(context, args);
                });

                // Check for user session invalidation
                if (hasUserSessionExpired(xhr)) {
                    // Allow time for the standard success/error callbacks to fire, so they don't overwrite the login page
                    _.defer(function () {
                        var auth = require('util/auth');
                        // If we're loading a resource that doesn't allow anonymous access, show the login page
                        if ('anonAllowed' in options && !options.anonAllowed) {
                            auth.showLoginPage();
                        // Otherwise, anon access is OK, so just trigger a logout event to update UI and settings
                        } else {
                            auth.triggerLogout();
                        }
                    });
                }
            };

            // Call Zepto AJAX method and track XHR
            var xhrPromise = $.ajax(options);
            currentXhrs.push(xhrPromise);

            // Handle aborted XHR properly. Zepto doesn't distinguish between an abort and a connection failure.
            var xhr = xhrPromise.origXHR;
            xhr.onabort = function () {
                // Set an "aborted" flag on the XHR to be handled by `handleAjaxError` later
                xhr.aborted = true;
            };

            return xhrPromise;
        }

        /**
         * Basic REST wrapper for `ajax()` - sets JSON headers
         *
         * @param options
         * @return {xhrPromise}
         */
        function rest(options) {
            options = $.extend({
                dataType: 'json',
                contentType: 'application/json',
//                headers: headers,
                jsonp: false,
                type : 'GET'
            }, options);

            return ajax(options);
        }

        exports.cancelActiveRequests = cancelActiveRequests;
        exports.ajax = ajax;
        exports.rest = rest;


        /*** ERROR HANDLING ***/

        /**
         * Check for user session invalidation by looking at a few response headers,
         * then comparing that with what we think we know about the user
         * and their authentication status.
         * @returns true if we're positive that the user was logged out before this XHR action completed.
         *   Otherwise, returns false (in all cases where we're not confident we know enough).
         */
        function hasUserSessionExpired(xhr) {
            var currentUser = String(appData.getUsername() || "").toLowerCase();

            function isUserLoggedIn() {
                var ANONYMOUS_USERNAMES = ["anonymous", "undefined", ""];
                return currentUser && _.contains(ANONYMOUS_USERNAMES, String(currentUser).toLowerCase()) === false;
            }

            if (isUserLoggedIn()) {
                // There's a logged-in user at the moment.
                var x_ausername = xhr.getResponseHeader('X-AUSERNAME');
                var usernameMismatch = x_ausername && (currentUser !== String(x_ausername).toLowerCase());

                var x_seraphLoginReason = xhr.getResponseHeader('X-Seraph-LoginReason');
                var badSeraphChallenge = x_seraphLoginReason && String(x_seraphLoginReason).toLowerCase() !== 'ok';

                return usernameMismatch || badSeraphChallenge;
            }
            // It's the anonymous user, so their session can't be expired.
            return false;
        }


        /**
         * Handle any global XHR errors that have not already been handled by the original caller.
         * Unless otherwise specified, this will show a full-page error.
         *
         * Individual AJAX error handlers can set several properties on the XHR object
         * to affect how this global handling works.
         *
         * e.g.
         * BackboneModel.fetch({
         *   error: function (model, xhr) {
         *     xhr.someproperty = somevalue;
         *   }
         * });
         *
         * List of properties:
         *
         *   xhr.aborted = true;  // Shouldn't be needed 99% of the time (already handled in this file)
         *   xhr.errorHandled = true;  // Bypasses this global error handling completely
         *   xhr.overrideStatus = httpStatusCode;  // For example, turn a 400 response into a 404
         */
        function handleAjaxError(xhr, settings, data) {
            router.presenter.hideLoading();
            if (xhr.aborted || xhr.errorHandled) {
                return;
            }
            var status = xhr.overrideStatus || xhr.status;

            // Handle authentication errors
            if (status === 401) {
                require('util/auth').showLoginPage();
                return;
            }

            var ErrorView = require('layout/error/error-view');
            var view = new ErrorView({
                status: status,
                url: settings.url,
                method: settings.type,
                data: data
            });
            router.presenter.showView('error' + status, view);
        };
    }
);
