define('util/tracking', ['backbone', 'underscore', 'zepto', 'util/app-data'], function (Backbone, _, $, appData) {

    /*** Config ***/

    // This are overridden if the appData key "analytics-enabled" is set to false
    const USE_ATLASSIAN_ANALYTICS = true;

    /*** Common Interface ***/

    const analyticsEnabled = appData.getBoolean('analytics-enabled');

    function elementToString(elem) {
        let str = elem.nodeName.toLowerCase();
        if (elem.id) {
            str += '#' + elem.id;
        } else {
            if (elem.classList.length) {
                str += '.' + _.toArray(elem.classList).join('.');
            }
        }
        return str;
    }

    const mappers = [];
    const tracker = {
        trackPageview: function (page) {
            _.each(mappers, function (mapper) {
                mapper.pageview(page);
            });
        },

        trackEvent: function (noun, verb, data) {
            _.each(mappers, function (mapper) {
                mapper.event(noun, verb, data);
            });
        },

        trackClick: function (element) {
            const selector = elementToString(element);
            this.trackEvent('element', 'click', {selector: selector});
        }
    };

    // Track all generic "clicks"
    $(function () {
        // Using old-school DOM method to bind on event capture phase instead of bubbling
        document.body.addEventListener('click', function (e) {
            let $target = $(e.target);
            // If element is a descendant of a link or button, track the link/button instead
            const $parent = $target.closest('a, button');
            if ($parent.length) {
                $target = $parent;
            }

            tracker.trackClick($target[0]);
        }, true);
    });

    // Track route changes
    Backbone.history.on('route', function () {
        tracker.trackPageview(Backbone.history.fragment);
    });

    /*** Atlassian Analytics ***/

    if (USE_ATLASSIAN_ANALYTICS && analyticsEnabled) {
        AJS.EventQueue = AJS.EventQueue || [];

        mappers.push({
            pageview: function (page) {
                this.event('page', 'view', {page: page});
            },
            event: function (noun, verb, data) {
                const name = ['mobile', noun, verb].join('.');
                const obj = {name: name};
                if (data) {
                    obj.properties = data;
                }
                // Stolen from jira-issue-nav-plugin/src/main/resources/content/js/util/ClientAnalytics.js
                let logMsg = "***** Analytics log [" + name + "]";
                if (data) {
                    logMsg += "[" + JSON.stringify(data) + "]";
                }
                AJS.log(logMsg);
                // Do the actual tracking
                AJS.EventQueue.push(obj);
            }
        });
    }

    return tracker;

});
