define('util/router',
    ['zepto', 'underscore', 'backbone', 'util/presenter', 'util/app-data', 'util/events', 'util/scroll-tops', 'require'],
    function ($, _, Backbone, presenter, appData, events, Scrolling, require) {
        const Router = Backbone.Router.extend({
            routes: {
                "login(/*fragment)": "login",
                "": "myOpenIssues",
                "myjirahome": "myOpenIssues",
                "issue/:issueKey(/comment/:commentId)": "viewIssue",
                "filter/assigned": "myOpenIssues",
                "filter/reported": "myReportedIssues",
                "filter/recent": "myRecentlyViewed",
                "filter/all": "allVisibleIssues",
                "filter/jql/:jql(/:filterId)": "jqlSearch",
                "filter/:id": "viewFilter",
                "filters/favourite": "favouriteFilters",
                "(*invalid)": "notFound"
            },

            initialize: function () {
                this.presenter = presenter.init({
                    container: $("#inner-container"),
                    cacheLength: 5
                });
                this.suppress = false;
                this.bind('all', function (event) {
                    const eventSplit = event.split(':');
                    if (eventSplit[0] == 'route') {
                        const args = _.toArray(arguments);
                        args[0] = eventSplit[1];
                        events.router.trigger.apply(events.router, ['change'].concat(args));
                    }
                });
            },

            goBackSilent: function () {
                this.suppress = true;
                window.history.back();
            },

            login: function (fragment) {
                const LoginView = require('layout/login/login-view');
                const options = {
                    cacheView: false,
                    clearCache: true
                };
                this.presenter.showView('login', new LoginView({
                    prevRoute: fragment
                }), options);
            },

            viewIssue: function (issueKey, commentId) {
                const cacheKey = 'viewIssue' + issueKey;
                const IssueView = require('feature/issues/issue-view');
                this.renderSingleIssue(issueKey, cacheKey, IssueView, commentId);
            },

            renderSingleIssue: function(issueKey, cacheKey, View, commentId) {
                const showBackButton = this.presenter.currentView && this.presenter.currentView.isList;
                if (this.presenter.currentView &&
                    this.presenter.currentView.model &&
                    this.presenter.currentView.model.get('key') === issueKey) {
                    // If the page we are currently on is the same issue as the page we're going to
                    // Just use that model instead of loading a new one. This method only works for the issue page and this should be redone when/if we do model caching for everything
                    const view = new View({
                        model: this.presenter.currentView.model,
                        showBackButton: showBackButton,
                        commentId: this.presenter.currentView.commentId
                    });
                    this.presenter.showView(cacheKey, view);
                } else {
                    // Load the issue and create a view from it
                    const IssueModel = require('feature/issues/issue-model');
                    const issue = new IssueModel({
                        key: issueKey
                    });
                    const that = this;
                    issue.fetch({
                        success: function () {
                            const view = new View({
                                model: issue,
                                showBackButton: showBackButton,
                                commentId: commentId
                            });
                            // If we are reloading the issue, then also reset the stored scroll position
                            Scrolling.resetCurrentHash();
                            that.presenter.showView(cacheKey, view);
                        }
                    });
                }
            },

            jqlSearch: function(jqlSearch, filterId) {
                const options = {
                    jql: decodeURIComponent(jqlSearch),
                    filterId: filterId,
                    title: AJS.I18n.getText('jira.mobile.filters.custom.jql')
                };

                this.renderIssues(false, options, {
                    isJql: true,
                    anonAllowed: true
                });
            },

            myOpenIssues: function () {
                const options = {
                    jql: 'assignee=currentUser() and resolution=Unresolved order by updated,priority',
                    filterId: -1,
                    title: AJS.I18n.getText('jira.mobile.filters.assigned.to.me')
                };

                this.renderIssues('myOpenIssues', options);
            },

            myReportedIssues: function () {
                const options = {
                    jql: 'reporter=currentUser() order by createdDate Desc',
                    filterId: -2,
                    title: AJS.I18n.getText('jira.mobile.filters.reported.by.me')
                };

                this.renderIssues('myReportedIssues', options);
            },

            myRecentlyViewed: function () {
                const options = {
                    jql: 'issue in issueHistory() order by lastViewed desc',
                    filterId: -3,
                    title: AJS.I18n.getText('jira.mobile.filters.recently.viewed')
                };

                this.renderIssues('myRecentlyViewed', options, {
                    anonAllowed: true
                });
            },

            allVisibleIssues: function () {
                const options = {
                    jql: 'ORDER BY createdDate DESC',
                    filterId: -4,
                    title: AJS.I18n.getText('jira.mobile.filters.all.visible')
                };

                this.renderIssues('allVisibleIssues', options, {
                    anonAllowed: true
                });
            },

            renderIssues: function (key, options, args) {
                if (!args) {
                    args = {};
                }
                // Don't show issue lists to anonymous users - they'll generally be empty... unless they came from a search, then being anonymous is allowed
                if (!args.anonAllowed && !appData.getUsername()) {
                    const auth = require('util/auth');
                    auth.showLoginPage();
                    return;
                }
                if (this.presenter.useCachedView(key)) {
                    return;
                }

                // Build the list of issues
                options.anonAllowed = !!args.anonAllowed;
                const IssueList = require('feature/issue-lists/issue-list-collection');
                const issueList = new IssueList([], options);
                var presenter = this.presenter;

                const IssueListView = require('feature/issue-lists/issue-list-view');
                const viewOptions = {
                    jql: args.isJql ? options.jql: '', // We only want to pass the jql to the view if this is a jql search
                    collection: issueList,
                    title: options.title,
                    filterId: options.filterId
                };
                let listView;

                function showList() {
                    listView = new IssueListView(viewOptions);
                    presenter.showView(key, listView);
                }

                issueList.fetch({
                    success: function () {
                        showList();
                    },
                    error: function (model, xhr) {
                        if (xhr.status == 400) {
                            let data = xhr.responseText;
                            if (data) {
                                try {
                                    data = JSON.parse(data);
                                } catch (e) {}
                            }
                            if (data.errorMessages) {
                                xhr.errorHandled = true;
                                viewOptions.error = data.errorMessages.join('\n');
                                // Make sure collection is empty
                                issueList.reset([], {silent: true});
                                showList();
                            }
                        }
                    }
                });
            },

            favouriteFilters: function () {
                // User must be logged in to view their favourite filters
                if (!appData.getUsername()) {
                    const auth = require('util/auth');
                    auth.showLoginPage();
                    return;
                }

                const cacheKey = 'favouriteFilters';
                if (this.presenter.useCachedView(cacheKey)) {
                    return;
                }

                // Build the list of issues
                const FilterList = require('feature/filters/filter-collection');
                const filters = new FilterList();
                var presenter = this.presenter;
                filters.fetch({
                    success: function () {
                        const FilterListView = require('feature/filters/filter-list-view');
                        const view = new FilterListView({
                            collection: filters,
                            title: AJS.I18n.getText('jira.mobile.filters.favourite')
                        });
                        presenter.showView(cacheKey, view);
                    }
                });
            },

            viewFilter: function (id) {
                const cacheKey = 'filter' + id;
                if (this.presenter.useCachedView(cacheKey)) {
                    return;
                }

                const favourites = require('feature/filters/favourites');
                const that = this;

                function showFilter(filter) {
                    const options = {
                        jql: filter.get('jql'),
                        filterId: id,
                        title: filter.get('name')
                    };

                    that.renderIssues(cacheKey, options, {
                        anonAllowed: true
                    });
                }

                // Make sure all favourite filters have loaded before looking for this one
                favourites.get(function (collection) {
                    let filter = favourites.collection.get(id);
                    // If the filter was in the user's favourites (the 95% case), show it immediately
                    if (filter) {
                        showFilter(filter);
                    // Otherwise we have to fetch the filter from the server
                    } else {
                        const Filter = require('feature/filters/filter-model');
                        filter = new Filter({id: id});
                        filter.fetch({
                            success: function () {
                                showFilter(filter);
                            },
                            error: function (model, xhr) {
                                // JIRA favourites API returns a 400 for an invalid ID, which should be a 404
                                if (xhr.status == 400) {
                                    xhr.overrideStatus = 404;
                                }
                            }
                        });
                    }
                });
            },

            notFound: function () {
                const ErrorView = require('layout/error/error-view');
                const view = new ErrorView({
                    status: 404
                });
                router.presenter.showView('error404', view);
            }
        });

        const router = new Router();

        router.navigate = _.wrap(router.navigate, function (_navigate) {
            const args = _.rest(arguments);
            // SAVE SCROLL HERE
            Scrolling.savePosition();
            return _navigate.apply(this, args);
        });

        /**
         * Override Backbone's hashchange event handler so we can silently go back one page
         * without re-triggering the route (used after cancelling a page load)
         */
        const _checkUrl = Backbone.history.checkUrl;
        Backbone.history.checkUrl = function () {
            if (!router.suppress) {
                _checkUrl.apply(this, arguments);
            } else {
                Backbone.history.fragment = Backbone.history.getFragment();
                router.suppress = false;
            }
        };

        // Add some classes for the browser/environment type.
        // Browser sniffing sucks but so does Android.
        const platformsWeCareAbout = ["ios", "android"],
              prefix = "platform-",
              platformsDetected = platformsWeCareAbout
                  .filter(function (platform) {
                      return $.os.hasOwnProperty(platform);
                  });

        if (platformsDetected.length) {
            $("body").addClass(prefix + platformsDetected.join(" " + prefix));
        }

        return router;
    }
);
