define('util/auth',
    ['backbone', 'util/ajax', 'util/app-data', 'util/events', 'util/router', 'util/scroll-tops'], // 'layout/login/login-view'
    function (Backbone, ajax, appData, events, router, Scrolling) {
        const contextPath = appData.get('context-path');

        /**
         * @note this is not be appropriate to use if an SSO solution is used.
         * @param {boolean} [clean=false] if true, no return destination or other GET parameters will be set.
         */
        const getInstanceLocalLoginUrl = function(clean) {
            let path = '/login.jsp';
            if (!clean) {
                const location = window.location;
                const currentPath = location.href.substr((location.origin + contextPath).length).replace('#login/', '#');
                path += '?os_destination=' + encodeURIComponent(currentPath);
            }
            return contextPath + path;
        };

        /**
         * @note this is not be appropriate to use if an SSO solution is used.
         */
        const getInstanceLocalLogoutUrl = function() {
            return contextPath + '/logout'
        };

        /**
         * @param {boolean} [clean=false] if true, no return destination or other GET parameters will be set.
         */
        const getSSOLoginUrl = function(clean) {
            return appData.get('sso-login-url');
        };

        const getSSOLogoutUrl = function() {
            return appData.get('sso-logout-url');
        };

        const auth = {
            hasSSO: function () {
                return appData.getBoolean('has-sso');
            },

            /**
             * @param {boolean} [clean=false] if true, the URL won't be adorned with additional GET parameters (other than
             *        the ones that may exist in the instance's Seraph-configured login URL or auth provider).
             * @returns {string} a root-relative or absolute URL for the login page.
             */
            getLoginDestination: function (clean) {
                return this.hasSSO() ? getSSOLoginUrl(clean) : getInstanceLocalLoginUrl(clean);
            },

            getLogoutDestination: function () {
                return this.hasSSO() ? getSSOLogoutUrl() : getInstanceLocalLogoutUrl();
            },

            showLoginPage: function (replace) {
                if (this.hasSSO()) {
                    return window.location = this.getLoginDestination();
                } else {
                    const fragment = Backbone.history.fragment;
                    if (fragment.substr(0, 6) != 'login/') {
                        // replace is true by default unless otherwise specified
                        const replaceVal = typeof replace === "undefined" ? true : !!replace;
                        router.navigate('login/' + fragment, {
                            trigger: true,
                            replace: replaceVal
                        });
                    }
                }
            },

            showCleanLoginPage: function () {
                if (this.hasSSO()) {
                    return window.location = this.getLoginDestination(true);
                } else {
                    router.navigate('login', {
                        trigger: true,
                        replace: false
                    });
                }
            },

            login: function (user) {
                this._updateUser(user);
                events.user.trigger('auth.login');
            },

            logout: function (options) {
                if (this.hasSSO()) {
                    return window.location = this.getLogoutDestination();
                } else {
                    this.triggerLogout();
                    options = options || {};
                    const willRedirect = options.showLoginPage !== false;
                    // We don't use a callback here because if the call fails with 401,
                    // the auth handler in util/ajax will show the login page anyway.
                    ajax.rest({
                        url: '/rest/auth/latest/session',
                        type: 'DELETE',
                        async: !willRedirect
                    });
                    if (options.showLoginPage !== false) {
                        this.showCleanLoginPage();
                    }
                }
            },

            /**
             * Clear the stored username and trigger the logout event.
             * Used for updating the UI when detecting a session timeout, but without showing the login page.
             */
            triggerLogout: function () {
                this._clearUser();
                Scrolling.resetAll();
                events.user.trigger('auth.logout');
            },

            _updateUser: function (user) {
                appData.put('remote-user', user.name);
                appData.put('user-data', user);
            },

            _clearUser: function () {
                appData.put('remote-user', '');
                appData.put('user-data', null);
            },

            on: function (event, handler) {
                events.user.on('auth.' + event, handler);
            },

            off: function (event, handler) {
                events.user.off('auth.' + event, handler);
            }
        };

        return auth;
    }
);
