define('feature/issues/comments/add-comment-view', [
    'zepto',
    'backbone',
    'underscore',
    'util/ajax',
    'feature/issues/comments/single-comment-model',
    'feature/issues/comments/single-comment-view',
    './comments.css',
    './comments.soy',
    'require'
],
function (
    $,
    Backbone,
    _,
    ajaxUtil,
    SingleCommentModel,
    SingleCommentView,
    css,
    Templates,
    require
) {
    require('widget/jquery-spinner/jquery-spinner');
    require('@atlassian/aui/entry/aui.component.message');

    const MESSAGES = {
        401: AJS.I18n.getText('jira.mobile.error.comments.logged.out'),
        'default': AJS.I18n.getText('jira.mobile.error.comments.failure')
    };

    return Backbone.View.extend({
        className: 'add-comment-area no-flicker',

        initialize: function (options) {
            this.postURL = '/comment/' + options.issueKey;
            this.submitting = false;
            this.highlightSecurityButton = $('body').hasClass('platform-ios');
        },

        events: {
            'submit form.add-comment': 'addComment',
            'focus .comment-value': 'focusComment',
            'input .comment-value': 'updateSubmitBtn',
            'click .at-mention': 'chooseUserMention',
            'click .comment-security': 'clickSecurity',
            'change .security-select': 'updatePadlock',
            'focus .security-select': 'focusSecurity',
            'blur .security-select': 'blurSecurity'
        },

        render: function () {
            this.$el.empty();
            this.$el.append(Templates.drawAddCommentForm({
                securityOptions: this.model.attributes.addCommentMeta
            }));
            this.$input = this.$('.comment-value');
            this.$securitySelect = this.$('.security-select');
            this.$securityButton = this.$('.comment-security');
            return this;
        },

        focusComment: function (e) {
            // Some versions of Android don't auto-scroll to the input field on focus,
            //  so it gets hidden behind the keyboard.
            if ($.os.android) {
                const top = this.$input.offset().top;
                // Need a setTimeout to allow time for the keyboard to finish showing
                setTimeout(function () {
                    window.scrollTo(0, top);
                }, 500);
            }
        },

        updateSubmitBtn: function (e) {
            if (!this.submitting) {
                const commentBody = this.$el.find('.comment-value').val();
                const submitButton = this.$el.find('.submit-comment');
                // Disable button if there is stuff in the comment box
                submitButton.attr('aria-disabled', !commentBody);
            }
        },

        clickSecurity: function () {
            this.$securitySelect.focus();
        },

        focusSecurity: function () {
            if (this.highlightSecurityButton) {
                this.$securityButton.addClass('active');
            }
        },

        blurSecurity: function () {
            if (this.highlightSecurityButton) {
                this.$securityButton.removeClass('active');
            }
        },

        updatePadlock: function (e){
            const currentSecurity = this.$el.find('.security-select').val();
            this.$el.find('.comment-security').toggleClass('selected',!!currentSecurity);
        },

        addComment: function(e) {
            e.preventDefault();
            const $errorContainer = this.$el.find('.add-error-container');
            const commentBodyBox = this.$el.find('.comment-value');
            const commentBody = commentBodyBox.val();
            // Don't bother sending nothing
            if (!commentBody) {
                return;
            }
            this.submitting = true;
            const data = {
                body: commentBody
            };
            this._addCommentSecurity(data);
            // Startup the spinner
            this._showSpinner();
            ajaxUtil.rest({
                type: 'POST',
                url: this.postURL,
                data: JSON.stringify(data),
                success: (data) => {
                    $errorContainer.empty();
                    this.submitting = false;
                    // Update the display
                    this._stopSpinner(data);
                },
                error: (xhr, errorType, error, data) => {
                    xhr.errorHandled = true;
                    // If the response didn't contain an error field, then let's assume it was a connection error
                    let message;
                    if (data.errors && data.errors.comment) {
                        message = data.errors.comment;
                    } else {
                        const key = data && data.reasonCode;
                        if (key) {
                            message = MESSAGES[key];
                        } else {
                            message = MESSAGES[xhr.status] || MESSAGES['default'];
                        }
                    }
                    $errorContainer.html(`<aui-message type="error">${message}</aui-message>`);
                    this.submitting = false;
                    this._stopSpinner(data, true);
                }
            });
        },

        _addCommentSecurity: function(data) {
            const value = this.$('.security-select').val();
            if (value) {
                const split = value.split(':');
                const visibility = {
                    type: split[0],
                    value: split[1]
                };
                data['visibility'] = visibility
            }
        },

        _showSpinner: function() {
            const commentButtons = this.$el.find('.add-comment-buttons');
            commentButtons.addClass('hidden');
            const loadingCancel = this.$el.find('.loading-cancel');
            loadingCancel.removeClass('hidden');
            const commentBox = this.$el.find('textarea.comment-value');
            commentBox.prop('disabled', true);
            // Start spinner
            const spinner = this.$el.find('.loading-cancel .spinner').spin('small', {
                left: '-42px',
                top: '2px'
            });
            // Store the elements so the stop method can just use them
            this.commentElements = {
                commentButtons: commentButtons,
                loadingCancel: loadingCancel,
                commentBox: commentBox,
                spinner: spinner
            }
        },

        _stopSpinner: function(data, isError) {
            // Stop the spinner
            this.commentElements.spinner.spinStop();
            this.commentElements.loadingCancel.addClass('hidden');
            const that = this;
            if (isError) {
                // For some reason, the button doesn't render otherwise...
                setTimeout( function () {
                    that.commentElements.commentButtons.removeClass('hidden');
                    that.commentElements.commentBox.prop('disabled', false);
                }, 0);
            } else {
                // Show success message
                const $success = this.$el.find('.loading-success');
                $success.addClass('show');
                setTimeout( function () {
                    // Fade the comment box
                    that.$el.find('.add-comment-container').animate({
                        opacity: 0
                    }, {
                        duration: 213,
                        easing: 'ease-in-out',
                        complete: function () {
                            // Finish up
                            that._finishAddComment(that, data);
                        }
                    });
                }, 750);
            }
        },

        chooseUserMention: function (e) {
            e.preventDefault();
            const searchUrl = '/rest/api/latest/user/viewissue/search?issueKey=' + encodeURIComponent(this.model.get('key')) + '&maxResults=10';
            const PickerView = require('widget/user-picker/overlay-view');
            const options = {
                titleText: AJS.I18n.getText('jira.mobile.issues.mention.title'),
                actionText: AJS.I18n.getText('jira.mobile.issues.mention.submit'),
                cancelText: AJS.I18n.getText('jira.mobile.issues.mention.cancel'),
                dataSrc: searchUrl,
                callback: _.bind(this.insertUserMention, this),
                cancelCallback: _.bind(this.cancelUserMention, this)
            };

            const view = new PickerView(options);
            require('util/router').presenter.showOverlay('atMention', view);

            // Reset the top offset of the overlay after the keyboard has hidden itself to avoid bad positioning - JMOB-290
            if ($.os.android) {
                setTimeout(function () {
                    view.refreshScrollTop();
                }, 400);
            }
        },

        insertUserMention: function (user) {
            if (!user) {
                this.cancelUserMention();
                return;
            }

            const wikiText = '[~' + user.get('name') + ']';
            const caretPos = this.$input[0].selectionStart;
            const prevVal = this.$input.val();
            const beforeText = prevVal.substr(0, caretPos);
            const afterText = prevVal.substr(caretPos);
            const beforeJoin = (!beforeText.length || beforeText.slice(-1) === ' ') ? '' : ' ';
            const afterJoin = afterText.substr(0, 1) === ' ' ? '' : ' ';

            const newVal = [beforeText, beforeJoin, wikiText, afterJoin, afterText].join('');
            const newCaretPos = caretPos + (beforeJoin + wikiText + afterJoin).length;
            var $input = this.$input;
            $input.prop('disabled', false).val(newVal).trigger('input').focus();
            $input[0].setSelectionRange(newCaretPos, newCaretPos);
        },

        cancelUserMention: function () {
            this.$input.focus();
        },

        _finishAddComment: function(context, data) {
            // Update the model
            const commentModel = new SingleCommentModel(data);
            commentModel.expand = true;
            context.model.addComment(new SingleCommentView({
                model: commentModel
            }));
            if (data.isAutoWatched) {
                context.model.attributes.permissions.watcherInfo.isWatching = true;
            }
            // Reset form
            context.render();
        }
    });
});
