define('util/tracking', ['underscore', 'zepto', 'util/app-data', 'util/auth', 'util/events'], function (_, $, appData, auth, events) {

    /*** Config ***/

    // This are overridden if the appData key "analytics-enabled" is set to false
    var USE_ATLASSIAN_ANALYTICS = true;
    var USE_GOOGLE_ANALYTICS = true;

    /*** Common Interface ***/

    var analyticsEnabled = appData.getBoolean('analytics-enabled');

    function elementToString(elem) {
        var str = elem.nodeName.toLowerCase();
        if (elem.id) {
            str += '#' + elem.id;
        } else {
            if (elem.classList.length) {
                str += '.' + _.toArray(elem.classList).join('.');
            }
            if (elem.href) {
                str += '[href=' + elem.getAttribute('href') + ']';
            }
        }
        return str;
    }

    var mappers = [];
    var tracker = {
        trackPageview: function (page) {
            _.each(mappers, function (mapper) {
                mapper.pageview(page);
            });
        },

        trackEvent: function (noun, verb, data) {
            _.each(mappers, function (mapper) {
                mapper.event(noun, verb, data);
            });
        },

        trackClick: function (element) {
            var selector = elementToString(element);
            this.trackEvent('element', 'click', {selector: selector});
        }
    };

    // Track all generic "clicks"
    $(function () {
        // Using old-school DOM method to bind on event capture phase instead of bubbling
        document.body.addEventListener('click', function (e) {
            var $target = $(e.target);
            // If element is a descendant of a link or button, track the link/button instead
            var $parent = $target.closest('a, button');
            if ($parent.length) {
                $target = $parent;
            }

            tracker.trackClick($target[0]);
        }, true);
    });

    // Track route changes
    Backbone.history.on('route', function () {
        tracker.trackPageview(Backbone.history.fragment);
    });

    /*** Google Analytics ***/

    if (USE_GOOGLE_ANALYTICS && analyticsEnabled) {
        window._gaq = window._gaq || [];
        window._gaq.push(['_setAccount', 'UA-20272869-16']);
        // Set username in a custom GA var (in slot 1) at the visitor-level scope
        window._gaq.push(['_setCustomVar', 1, 'username', appData.getUsername(), 1]);

        (function() {
            var ga = document.createElement('script'); ga.type = 'text/javascript'; ga.async = true;
            ga.src = ('https:' == document.location.protocol ? 'https://ssl' : 'http://www') + '.google-analytics.com/ga.js';
            var s = document.getElementsByTagName('script')[0]; s.parentNode.insertBefore(ga, s);
        })();

        mappers.push({
            pageview: function (page) {
                window._gaq.push(['_trackPageview', page]);
            },
            event: function (name, verb, data) {
                var args = ['_trackEvent', name, verb];
                if (data) {
                    var label = data.selector;
                    if (label) {
                        args.push(label);
                    }
                }
                window._gaq.push(args);
            }
        });

        auth.on('login', function () {
            window._gaq.push(['_setCustomVar', 1, 'username', appData.getUsername(), 1]);
        });
        auth.on('logout', function () {
            window._gaq.push(['_setCustomVar', 1, 'username', '', 1]);
        });
    }

    /*** Atlassian Analytics ***/

    if (USE_ATLASSIAN_ANALYTICS && analyticsEnabled) {
        AJS.EventQueue = AJS.EventQueue || [];

        mappers.push({
            pageview: function (page) {
                this.event('page', 'view', {page: page});
            },
            event: function (noun, verb, data) {
                var name = ['mobile', noun, verb].join('.');
                var obj = {name: name};
                if (data) {
                    obj.properties = data;
                }
                // Stolen from jira-issue-nav-plugin/src/main/resources/content/js/util/ClientAnalytics.js
                var logMsg = "***** Analytics log [" + name + "]";
                if (data) {
                    logMsg += "[" + JSON.stringify(data) + "]";
                }
                AJS.log(logMsg);
                // Do the actual tracking
                AJS.EventQueue.push(obj);
            }
        });
    }

    return tracker;

});