var webhooks = function () {
  // eslint-disable-line
  'use strict';

  var Backbone = require('webhooks/lib/backbone');
  var TEMPLATES = atl.plugins.webhooks.admin.templates; // eslint-disable-line

  var events = _.extend({}, Backbone.Events); // eslint-disable-line
  var sections = [];
  var dateFormatter = function dateFormatter(date) {
    return date.toLocaleDateString() + " " + date.toLocaleTimeString();
  };
  var WebHookModel = Backbone.Model.extend({
    url: function url() {
      return this.collection.url + (this.get("id") ? '/' + this.get("id") : '');
    },
    defaults: {
      name: AJS.I18n.getText('webhooks.add.newwebhookname'),
      id: null,
      url: 'http://example.com/rest/webhooks/webhook1',
      events: [],
      active: true,
      scopeType: "global",
      configuration: {
        FILTERS: "",
        DESCRIPTION: "",
        EXCLUDE_BODY: 'false',
        SECRET: ""
      }
    },
    idAttribute: "id"
  });
  var WebHooksCollection = Backbone.Collection.extend({
    url: AJS.contextPath() + '/rest/jira-webhook/1.0/webhooks',
    model: WebHookModel,
    comparator: function comparator(left, right) {
      return left.get("name").localeCompare(right.get("name"));
    },
    initialize: function initialize() {
      this.on("change:name", function () {
        this.sort();
      }, this);
    }
  });
  var SelectionModel = Backbone.Model.extend({
    select: function select(model) {
      if (this.isLocked()) {
        var answer = confirm(AJS.I18n.getText("webhooks.leave.page.confirmation"));
        if (!answer) {
          return;
        }
      }
      this.set("selection", model);
    },
    getSelected: function getSelected() {
      return this.get("selection");
    },
    lock: function lock() {
      this.set("locked", true);
    },
    unlock: function unlock() {
      this.unset("locked");
    },
    isLocked: function isLocked() {
      return this.has("locked");
    },
    onSelectionChange: function onSelectionChange(callback, that) {
      this.on("change:selection", callback, that);
    },
    onLockChange: function onLockChange(callback, that) {
      this.on("change:locked", callback, that);
    }
  });
  var WebHookRow = Backbone.View.extend({
    tagName: 'li',
    events: {
      "click": "rowClicked"
    },
    initialize: function initialize(options) {
      this.options = options;
      this.selectionModel = this.options.selectionModel;
      this.selectionModel.onSelectionChange(this.selectionChanged, this);
      this.model.on("remove", this.modelRemoved, this);
      this.model.on("change", this.change, this);
    },
    render: function render() {
      this.$el.html(TEMPLATES.webhookRow(this.model.attributes));
      this.$el.toggleClass("webhook-disabled", !this.model.get("active"));
      this.selectionChanged(this.selectionModel, this.selectionModel.getSelected()); // handle event handlers reorder
      return this;
    },
    change: function change() {
      this.$el.toggleClass("webhook-disabled", !this.model.get("active"));
    },
    rowClicked: function rowClicked() {
      this.selectionModel.select(this.model);
    },
    selectionChanged: function selectionChanged(selectionModel, selectedModel) {
      this.$el.toggleClass("highlighted", selectedModel === this.model);
    },
    modelRemoved: function modelRemoved() {
      this.destroy();
      this.remove();
    },
    destroy: function destroy() {
      this.undelegateEvents();
      this.selectionModel.off(null, null, this);
      this.model.off(null, null, this);
    }
  });
  var WebHookDetailsView = Backbone.View.extend({
    el: ".webhook-details",
    events: {
      "submit form": "submit",
      "click #webhook-edit": "editMode",
      "click #webhook-delete": "deleteWebhook",
      "click #webhook-disabled": "toggleEnable",
      "click #webhook-enabled": "toggleEnable",
      "click #webhook-cancel": "cancel"
    },
    initialize: function initialize() {
      var $el = this.$el;
      this.statusEnabled = true;
      this.$form = $el.find("form");
      this.$globalGessage = $el.find("#webhook-global-message");
      this.$enabledLozenge = this.$form.find('#webhook-enabled-lozenge');
      this.$disabledLozenge = this.$form.find('#webhook-disabled-lozenge');
      this.$editedDate = $el.find("#webhook-edited-date");
      // this.$editedBy = $el.find("#webhook-edited-by");
      this.$name = $el.find("#webhook-name");
      this.$nameDisplay = $el.find("#webhook-name-display");
      this.$enabled = $el.find("#webhook-enabled");
      this.$disabled = $el.find("#webhook-disabled");
      this.$url = $el.find("#webhook-url");
      this.$urlDisplay = $el.find("#webhook-url-display");
      this.$description = $el.find("#webhook-description");
      this.$descriptionDisplay = $el.find("#webhook-description-display");
      this.$secret = $el.find("#webhook-secret");
      this.$secretDisplay = $el.find("#webhook-secret-display");
      this.$eventCheckboxes = $el.find(".webhook-checkbox");
      this.$noEventsSelected = $el.find("#webhook-no-events-selected");
      this.$sectionDisplay = $el.find(".display-mode-display .webhook-section");
      this.$groupDisplay = $el.find(".display-mode-display .webhook-group");
      this.$eventDisplay = $el.find(".display-mode-display .webhook-group li");
      this.$excludeDetails = $el.find("#webhook-exclude-details");
      this.$excludeDetailsDisplay = $el.find("#webhook-exclude-details-display");
      this.$submit = this.$form.find('#webhook-submit');
      this.selectedModel = undefined;
      this.selectionModel = this.model;
      this.selectionModel.onSelectionChange(this.selectionChanged, this);
      this.$form.on("cancel", function () {
        return false;
      }); // so dirty form warning works after cancel

      events.trigger("sections:init", this.selectionModel);
      var $ = AJS.$;
      this.renderAvailableVariables = function () {
        var variablesMap = {};
        $(".webhook-event").find("input:checked").each(function () {
          variablesMap[$(this).attr("data-event-type")] = $(this).attr("urlVariables").split(",");
        });
        var sum = _.uniq(_.flatten(_.values(variablesMap))); // eslint-disable-line
        var intersection = _.reduce(_.values(variablesMap), function (memo, newArray) {
          // eslint-disable-line
          return _.intersection(memo, newArray); // eslint-disable-line
        }, sum);
        var other = _.difference(sum, intersection); // eslint-disable-line

        $("li.variable").each(function () {
          $(this).removeClass("variable-intersection");
          $(this).removeClass("variable-some");
          $(this).addClass("variable-unavailable");
          $(this).attr("title", AJS.I18n.getText("webhooks.variables.usage.none"));
          if (_.indexOf(intersection, $(this).attr("variable")) !== -1) {
            // eslint-disable-line
            $(this).addClass("variable-intersection");
            $(this).removeClass("variable-unavailable");
            $(this).attr("title", AJS.I18n.getText("webhooks.variables.usage.all"));
          }
          if (_.indexOf(other, $(this).attr("variable")) !== -1) {
            // eslint-disable-line
            $(this).addClass("variable-some");
            $(this).removeClass("variable-unavailable");
            $(this).attr("title", AJS.I18n.getText("webhooks.variables.usage.some"));
          }
        });
      };
      var urlInput = this.$url;
      $el.find("[urlVariables]").change(this.renderAvailableVariables);
      $el.find(".variable").click(function () {
        var variable = $(this).text();
        urlInput.val(urlInput.val() + variable);
        urlInput.focus();
      });
    },
    render: function render() {
      var renderSectionFilters = function renderSectionFilters(filters) {
        _.each(_.pairs(sections), function (section) {
          // eslint-disable-line
          section[1].setFilter(filters || "");
        });
      };
      var that = this;

      // this setting the height of the webhooks navigation to 100% of webhook section height
      var sectionHeight = AJS.$("section.aui-page-panel-content").outerHeight();
      var headerHeight = AJS.$(".webhooks .aui-page-header").outerHeight();
      AJS.$(".webhooks .aui-page-panel-nav").height(sectionHeight - headerHeight);
      this.$form.find(".error").empty();
      this.$el.find(".1buttons-container").toggle(!!this.selectedModel);
      if (this.selectedModel) {
        var model = this.selectedModel;
        this.statusEnabled = model.get("active");
        if (this.statusEnabled) {
          this.$enabledLozenge.show();
          this.$disabledLozenge.hide();
        } else {
          this.$enabledLozenge.hide();
          this.$disabledLozenge.show();
        }

        // this.$editedBy.text(model.get("lastUpdatedDisplayName"));

        var rawUpdatedDate = model.get("updatedDate");
        var updatedDate = new Date(rawUpdatedDate);
        var updatedDateString = dateFormatter(updatedDate);
        this.$editedDate.html(updatedDateString);
        this.$name.val(model.get("name"));
        this.$nameDisplay.text(model.get("name"));
        this.$url.val(model.get("url"));
        this.$urlDisplay.text(model.get("url")).attr({
          href: model.get("url")
        });
        this.$description.val(model.get("configuration")["DESCRIPTION"]);
        this.$descriptionDisplay.text(model.get("configuration")["DESCRIPTION"]);
        this.$secret.val(model.get("configuration")["SECRET"]);
        this.$secretDisplay.text(model.get("configuration")["SECRET"]);
        this.renderStatusButtons();
        this.renderEvents();
        this.$excludeDetails.prop("checked", model.get("configuration")["EXCLUDE_BODY"] === "true");
        var excludeDetailsDisplayText = model.get("configuration")["EXCLUDE_BODY"] === "true" ? AJS.I18n.getText('common.words.yes') : AJS.I18n.getText('common.words.no');
        this.$excludeDetailsDisplay.text(excludeDetailsDisplayText);
        renderSectionFilters(model.get("configuration")["FILTERS"], this.selectionModel);
        this.renderAvailableVariables();
        this.$submit.val(model.isNew() ? AJS.I18n.getText('webhooks.create') : AJS.I18n.getText('webhooks.save'));
        this.$el.find('input.text, textarea').each(function (idx, el) {
          el.defaultValue = el.value;
        });
        this.$globalGessage.empty();
      } else {
        this.$name.val("");
        this.$nameDisplay.text("");
        this.$url.val("");
        this.$urlDisplay.text("");
        this.statusEnabled = true;
        this.$eventCheckboxes.each(function () {
          that.$(this).prop("checked", false);
        });
        renderSectionFilters("", this.selectionModel);
      }
    },
    renderStatusButtons: function renderStatusButtons() {
      this.$enabled.toggleClass("active", this.statusEnabled);
      this.$enabled.prop("disabled", this.statusEnabled);
      this.$disabled.toggleClass("active", !this.statusEnabled);
      this.$disabled.prop("disabled", !this.statusEnabled);
    },
    renderEvents: function renderEvents() {
      var that = this;
      var events = this.selectedModel.get("events");
      this.$noEventsSelected.css("display", events.length === 0 ? "" : "none");
      this.$eventCheckboxes.each(function () {
        var $checkbox = that.$(this);
        $checkbox.prop("checked", _.indexOf(events, $checkbox.attr('data-event-type')) !== -1); // eslint-disable-line
      });
      this.$eventDisplay.removeClass("show");
      this.$eventDisplay.removeClass("last");
      this.$eventDisplay.each(function () {
        var $event = that.$(this);
        var exist = _.indexOf(events, $event.attr('data-event-type')) !== -1; // eslint-disable-line
        if (exist) {
          $event.addClass("show");
        }
      });
      this.$groupDisplay.removeClass("show");
      this.$groupDisplay.each(function () {
        var $group = that.$(this);
        var events = $group.find("li.show");
        if (events.size() > 0) {
          $group.addClass("show");
          events.last().addClass("last");
        }
      });
      this.$sectionDisplay.removeClass("show");
      this.$sectionDisplay.each(function () {
        var $section = that.$(this);
        if ($section.find(".webhook-group.show").size() > 0) {
          $section.addClass("show");
        }
      });
    },
    editMode: function editMode() {
      this.selectionModel.lock();
      this.$form.addClass("display-mode-edit").removeClass("display-mode-display");
      this.$noEventsSelected.css("display", "none");
    },
    displayMode: function displayMode() {
      this.selectionModel.unlock();
      this.$form.addClass("display-mode-display").removeClass("display-mode-edit");
    },
    toggleEnable: function toggleEnable() {
      this.statusEnabled = !this.statusEnabled;
      this.renderStatusButtons();
      return false;
    },
    getEvents: function getEvents() {
      var events = [];
      var that = this;
      this.$(".webhook-event").find("input:checked").each(function () {
        events.push(that.$(this).attr("data-event-type"));
      });
      return events;
    },
    submit: function submit() {
      var getFilters = function getFilters() {
        var result = {};
        _.each(_.pairs(sections), function (section) {
          // eslint-disable-line
          result[section[0]] = section[1].getFilter();
        });
        return result;
      };
      this.$form.find('.error').empty();
      var that = this;
      var wasNew = this.selectedModel.isNew();
      var configValues = new Map();
      if (getFilters()["issue-related-events-section"].length) {
        configValues["FILTERS"] = getFilters()["issue-related-events-section"];
      }
      if (this.$description.val().length) {
        configValues["DESCRIPTION"] = this.$description.val();
      }
      if (this.$secret.val().length) {
        configValues["SECRET"] = this.$secret.val();
      }
      configValues["EXCLUDE_BODY"] = this.$excludeDetails.is(":checked").toString();
      this.selectedModel.save({
        name: this.$name.val(),
        url: this.$url.val(),
        events: this.getEvents(),
        active: this.statusEnabled,
        scopeType: "global",
        configuration: configValues
      }, {
        wait: true,
        success: submitSuccess,
        error: submitError
      });
      return false; // so dirty form warning works after first submit

      function scrollTop() {
        AJS.$('html, body').animate({
          scrollTop: AJS.$('.webhooks').offset().top
        }, 400);
      }
      function submitSuccess(model) {
        var successMessage;
        if (wasNew) {
          successMessage = AJS.I18n.getText("webhooks.create.success", model.get("name"));
        } else {
          successMessage = AJS.I18n.getText("webhooks.update.success", model.get("name"));
        }
        displaySuccessMessage(successMessage);
        that.displayMode();
        scrollTop();
      }
      function submitError(model, response) {
        try {
          var errorObject = AJS.$.parseJSON(response.responseText);
        } catch (parseError) {
          errorObject = {};
        }
        if (response.status === 404) {
          displayErrorMessage(AJS.I18n.getText("webhooks.submit.notfound"));
        } else if (response.status === 400 || response.status === 409) {
          var fieldErrors = errorObject.messages || {};
          _.each(fieldErrors, function (error) {
            // eslint-disable-line
            if (error.arguments) {
              that.$el.find("#webhook-" + error.key).siblings(".error").text(error.arguments[0]);
            } else {
              AJS.messages.error(that.$globalGessage, {
                title: AJS.I18n.getText("webhooks.submit.duplicate.title"),
                body: "<p>" + AJS.escapeHtml(error.key) + "</p>",
                closeable: false
              });
            }
          });
        } else {
          displayErrorMessage(AJS.I18n.getText("webhooks.submit.error", response.status, response.statusText));
        }
        scrollTop();
      }
    },
    cancel: function cancel() {
      this.displayMode();
      if (this.selectedModel.isNew()) {
        this.selectedModel.destroy();
      } else {
        this.render();
      }
    },
    deleteWebhook: function deleteWebhook() {
      this.selectedModel && _deleteWebhook(this.selectedModel);
    },
    selectionChanged: function selectionChanged(selectionModel, selectedModel) {
      if (this.selectedModel && this.selectedModel.isNew()) {
        this.selectedModel.destroy();
      }
      this.displayMode();
      if (this.selectedModel) {
        this.selectedModel.off(null, null, this); // unbind any listeners created by this view
      }
      this.selectedModel = selectedModel;
      if (this.selectedModel) {
        this.selectedModel.on("change", this.render, this);
        selectedModel.isNew() && this.editMode();
      }
      this.render();
    }
  });
  var WebHooksNav = Backbone.View.extend({
    tagName: 'ul',
    className: 'aui-nav',
    initialize: function initialize(options) {
      this.options = options;
      this.model.on('add', this.modelAdded, this);
      this.model.on('reset', this.reset, this);
      this.model.on('change', this.reset, this);
      this.$rowsCollection = [];
    },
    reset: function reset() {
      this.$el.empty();
      _.each(this.$rowsCollection, function (webHookRow) {
        // eslint-disable-line
        webHookRow.destroy();
      });
      this.$rowsCollection = [];
      var that = this;
      this.model.each(function (model) {
        var webHookRow = new WebHookRow({
          model: model,
          selectionModel: that.options.selectionModel
        });
        that.$el.append(webHookRow.render().$el);
        that.$rowsCollection.push(webHookRow);
      });
    },
    render: function render() {
      return this;
    },
    modelAdded: function modelAdded(model) {
      var webHookRow = new WebHookRow({
        model: model,
        selectionModel: this.options.selectionModel
      });
      this.$el.prepend(webHookRow.render().$el);
      this.$rowsCollection.push(webHookRow);
    }
  });
  var AddWebhookButton = Backbone.View.extend({
    events: {
      "click": "addWebhook"
    },
    initialize: function initialize(options) {
      this.options = options;
      this.model.onLockChange(this.render, this);
    },
    render: function render() {
      this.$el.prop("disabled", this.model.isLocked());
    },
    addWebhook: function addWebhook() {
      !this.model.isLocked() && this.options.webHooksModel.add({/* defaults*/}, {
        at: 0
      });
      return false;
    }
  });
  var GlobalPageView = Backbone.View.extend({
    el: ".webhooks",
    initialize: function initialize() {
      this.model.on('reset add remove request', this.countChanged, this);
    },
    countChanged: function countChanged() {
      var empty = this.model.length === 0;
      this.$el.find(".on-webhooks-absent").toggleClass("hidden", !empty);
      this.$el.find(".on-webhooks-present").toggleClass("hidden", empty);
    }
  });
  var webHooksModel = new WebHooksCollection();
  var selectionModel = new SelectionModel();
  webHooksModel.on('remove', function (removedModel) {
    if (selectionModel.getSelected() === removedModel) {
      selectionModel.select(webHooksModel.at(0));
    }
  });
  webHooksModel.on('add', function (model) {
    selectionModel.select(model);
  });
  AJS.$(function () {
    // eslint-disable-line
    new AddWebhookButton({
      model: selectionModel,
      el: "#add-webhook",
      webHooksModel: webHooksModel
    });
    new WebHookDetailsView({
      model: selectionModel
    }).render();
    AJS.$(".aui-navgroup-inner").append(new WebHooksNav({
      model: webHooksModel,
      selectionModel: selectionModel
    }).render().$el);
    new GlobalPageView({
      model: webHooksModel
    });
    webHooksModel.fetch({
      error: function error(model, response) {
        displayErrorMessage(AJS.I18n.getText("webhooks.fetch.error", response.status, response.statusText));
      },
      success: function success(model) {
        var id = window.location.hash;
        if (id && id !== "") {
          selectionModel.select(model.get(parseInt(id.substring(1))));
        } else {
          selectionModel.select(model.at(0));
        }
      }
    });
    AJS.$("#webhook-submit, #webhook-edit").click(function () {
      AJS.$("#webhook-global-message").empty();
    });
  });
  function _deleteWebhook(model) {
    if (selectionModel.isLocked()) {
      return;
    }
    var popup = new AJS.Dialog();
    function destroyModel() {
      popup.remove();
      model.destroy({
        wait: true,
        success: function success(model) {
          displaySuccessMessage(AJS.I18n.getText("webhooks.delete.success", model.get("name")));
        },
        error: function error(model, response) {
          if (response.status === 409) {
            var errorResponse = AJS.$.parseJSON(response.responseText);
            var messages = errorResponse.messages;
            displayErrorMessageNoEscape(_.pluck(messages, 'arguments').join("<br />")); // eslint-disable-line
          } else {
            displayErrorMessage(AJS.I18n.getText("webhooks.delete.error", model.get("name"), response.status, response.statusText));
          }
        }
      });
    }
    popup.addHeader(AJS.I18n.getText('webhooks.delete.title')).addPanel('warning-message', AJS.$('<p>').text(AJS.I18n.getText('webhooks.delete.confirm', model.get('name')))).addButton(AJS.I18n.getText('webhooks.delete'), destroyModel, 'aui-button').addCancel(AJS.I18n.getText('webhooks.cancel'), function () {
      popup.remove();
    }).show().updateHeight();
  }
  function displaySuccessMessage(message) {
    AJS.messages.success(AJS.$("#webhook-global-message"), {
      body: AJS.escapeHtml(message),
      closeable: true
    });
  }
  function displayErrorMessage(message) {
    AJS.messages.error(AJS.$("#webhook-global-message"), {
      body: AJS.escapeHtml(message),
      closeable: true
    });
  }
  function displayErrorMessageNoEscape(message) {
    AJS.messages.error(AJS.$("#webhook-global-message"), {
      body: message,
      closeable: true
    });
  }
  return {
    addFilter: function addFilter(sectionName, getFilter, setFilter) {
      sections[sectionName] = {
        getFilter: getFilter,
        setFilter: setFilter
      };
    },
    setDateFormatter: function setDateFormatter(customDateFormatter) {
      dateFormatter = customDateFormatter;
    },
    onInit: function onInit(callback) {
      events.on("sections:init", callback);
    }
  };
}();